<?php
declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

// ===== Cabeceras (API JSON + CORS para pruebas) =====
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type');
header('Access-Control-Allow-Methods: POST, OPTIONS');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

// ===== Manejo de errores en JSON (útil para debug) =====
ini_set('display_errors', '0');
error_reporting(E_ALL);
set_error_handler(function($s,$m,$f,$l){
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'PHP error','detail'=>"$m in $f:$l"], JSON_UNESCAPED_UNICODE);
  exit;
});
set_exception_handler(function($e){
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Exception','detail'=>$e->getMessage()], JSON_UNESCAPED_UNICODE);
  exit;
});

// ===== Conexión =====
// Ajusta la ruta si tu conex.php está en otro lado.
// En tu proyecto sueles tenerlo en views/pages/conex.php
require_once __DIR__ . '/../conex.php'; // Debe definir $conn (mysqli)

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Método no permitido']); exit;
}

// ===== Entradas =====
$in = json_decode(file_get_contents('php://input'), true);
if (!is_array($in) || empty($in)) $in = $_POST; // fallback para form-data

$usuario = trim($in['usuario'] ?? '');
$pass    = (string)($in['pass'] ?? '');

if ($usuario === '' || $pass === '') {
  http_response_code(400);
  echo json_encode(['success'=>false,'message'=>'Usuario y contraseña requeridos']); exit;
}

// ===== Login =====
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
$stmt = $conn->prepare(
  "SELECT 
      idusuario AS id, usuario, clave,
      id_rol     AS permiso,
      id_registro, id_bodega, id_empresa,
      activo, condicion
   FROM tbr_usuario
   WHERE usuario = ? LIMIT 1"
);
$stmt->bind_param('s', $usuario);
$stmt->execute();
$u = $stmt->get_result()->fetch_assoc();

if (!$u || (int)$u['activo'] !== 1 || (int)$u['condicion'] !== 1) {
  http_response_code(401);
  echo json_encode(['success'=>false,'message'=>'Credenciales inválidas']); exit;
}

// Verifica contraseña (recomendado: hash con password_hash)
$ok = password_verify($pass, $u['clave']);
// Si aún tienes claves sin hash (no recomendado), habilita el fallback temporal:
// $ok = $ok || $pass === $u['clave'];

if (!$ok) {
  http_response_code(401);
  echo json_encode(['success'=>false,'message'=>'Credenciales inválidas']); exit;
}

// ===== Respuesta =====
// (Solo login: devolvemos datos del usuario. Si luego quieres token, te lo agrego.)
echo json_encode([
  'success' => true,
  'user' => [
    'id'          => (int)$u['id'],
    'usuario'     => $u['usuario'],
    'permiso'     => (int)$u['permiso'],
    'id_registro' => (int)($u['id_registro'] ?? 0),
    'id_bodega'   => (int)($u['id_bodega'] ?? 0),
    'id_empresa'  => (int)($u['id_empresa'] ?? 0),
  ]
], JSON_UNESCAPED_UNICODE);
