<?php
// factura_informativa_pdf.php — versión optimizada
declare(strict_types=1);

require '../views/pages/conex.php';
require 'fpdf/fpdf.php';

// ===== Configuración base =====
date_default_timezone_set('America/Guayaquil');
mysqli_set_charset($conn, 'utf8mb4');

// Sanitizar entrada
$idfactura = isset($_GET['idfactura']) ? (int)$_GET['idfactura'] : 0;
if ($idfactura <= 0) {
  http_response_code(400);
  exit('Parámetro idfactura inválido');
}

// ===== Helpers =====
function toLatin1(string $s): string {
  // FPDF espera ISO-8859-1; si hay caracteres fuera de rango, los reemplaza
  return mb_convert_encoding($s, 'ISO-8859-1', 'UTF-8');
}

function tipoIdentificacion(string $numero): string {
  $len = strlen($numero);
  if ($len === 10) return 'CEDULA IDENTIDAD';
  if ($len === 13) return 'RUC';
  return 'PASAPORTE';
}

function fmtNum(float $n, int $dec = 3): string {
  return number_format($n, $dec, '.', ',');
}

// Fecha en español robusta (Intl si está disponible)
function fechaLarga(string $ymd): string {
  if (class_exists('IntlDateFormatter')) {
    $fmt = new IntlDateFormatter('es_EC', IntlDateFormatter::FULL, IntlDateFormatter::NONE, 'America/Guayaquil', null, "EEEE, d 'de' MMMM 'de' y");
    $ts = strtotime($ymd);
    return $fmt->format($ts);
  }
  // Fallback simple
  $dias = ["domingo","lunes","martes","miércoles","jueves","viernes","sábado"];
  $meses = ["enero","febrero","marzo","abril","mayo","junio","julio","agosto","septiembre","octubre","noviembre","diciembre"];
  $ts = strtotime($ymd);
  return ucfirst($dias[(int)date('w', $ts)]) . ', ' . date('d', $ts) . ' de ' . $meses[(int)date('n', $ts)-1] . ' de ' . date('Y', $ts);
}

// ===== 1) Actualizar estado de cabecera =====
$upd = $conn->prepare("UPDATE tb_facturainfo SET id_estado = 6 WHERE idfacturainfo = ?");
$upd->bind_param('i', $idfactura);
$upd->execute();
// No abortar si no afectó filas; puede ya estar en ese estado

// ===== 2) Consultar cabecera y metadatos =====
// - Si exonerado = 0, usar datos de tb_empresa; si no, usar fi.cedula y fi.nombre
// - Moneda viene desde solicitud previa (sp.id_moneda) y sus siglas/detalle desde tb_moneda
$sqlCab = $conn->prepare("
  SELECT
    fi.idfacturainfo, fi.numero, fi.fecha AS fecha_fi, fi.exonerado,
    fi.cedula AS cedula_fi, fi.nombre AS nombre_fi,
    sp.cod_embarque, sp.numerosp AS sp, sp.pedido, sp.fecha AS spfecha,
    de.numerodai AS dai, de.fecha AS fecha_regi,
    m.siglas AS moneda_siglas, m.detalle AS moneda_detalle,
    e.ruc AS ruc_emp, e.razon_social
  FROM tb_facturainfo AS fi
  INNER JOIN tb_detallefactura  AS df ON df.id_facturainfo = fi.idfacturainfo
  INNER JOIN tb_datadai         AS d  ON d.iddatadai = df.id_datadai
  INNER JOIN tb_solicitudprevia AS sp ON sp.idsolicitudprevia = d.id_solicitudprevia
  INNER JOIN tb_daientrada      AS de ON de.iddaientrada = d.id_daientrada
  LEFT  JOIN tb_moneda          AS m  ON m.idmoneda = sp.id_moneda
  LEFT  JOIN tb_empresa         AS e  ON e.idempresa = fi.id_cliente
  WHERE fi.idfacturainfo = ? AND fi.condicion = 1
  LIMIT 1
");
$sqlCab->bind_param('i', $idfactura);
$sqlCab->execute();
$head = $sqlCab->get_result()->fetch_assoc();

if (!$head) {
  http_response_code(404);
  exit('No se encontró la cabecera de la factura.');
}

// Variables cabecera
$exonerado   = (int)$head['exonerado'];
$numero      = (string)$head['numero'];
$fechaFI     = (string)$head['fecha_fi'];      // Y-m-d
$pedido      = (string)$head['pedido'];
$dai         = (string)$head['dai'];
$sp          = (string)$head['sp'];
$fechaSP     = (string)$head['spfecha'];       // Y-m-d
$fechaReg    = (string)$head['fecha_regi'];    // Y-m-d
$monSiglas   = (string)($head['moneda_siglas'] ?? '');
$monDetalle  = (string)($head['moneda_detalle'] ?? '');
$embarque    = (string)$head['cod_embarque'];

// Cliente/identificación
if ($exonerado === 0) {
  $nombreCliente = (string)$head['razon_social'];
  $identificacion = (string)$head['ruc_emp'];
} else {
  $nombreCliente = (string)$head['nombre_fi'];
  $identificacion = (string)$head['cedula_fi'];
}
$tipoId = tipoIdentificacion($identificacion);

// Fecha formateada
$fechaSP_Larga = fechaLarga($fechaSP);

// ===== 3) Detalle de ítems =====
$sqlDet = $conn->prepare("
  SELECT df.item, df.id_datadai,
         fd.marca, fd.descripcion, fd.modelo, fd.color, fd.otras, fd.chasis, fd.motor,
         fd.cantidad, fd.fobitem, fd.seguro, fd.transporte
  FROM tb_detallefactura AS df
  INNER JOIN tb_datadai AS fd ON fd.iddatadai = df.id_datadai
  WHERE df.id_facturainfo = ? AND df.condicion = 1
  ORDER BY df.item ASC
");
$sqlDet->bind_param('i', $idfactura);
$sqlDet->execute();
$detRes = $sqlDet->get_result();

$items = [];
while ($r = $detRes->fetch_assoc()) {
  $items[] = $r;
}

if (!$items) {
  http_response_code(404);
  exit('La factura no tiene ítems.');
}

// ===== 4) Preparar PDF =====
$pdf = new FPDF('L', 'mm', 'Legal');
$pdf->AddPage();
$pdf->AliasNbPages();

// Marco superior
$pdf->SetXY(85, 5);
$pdf->Cell(258, 28, "", 1);

// Logo + bloque izquierdo
$pdf->SetFont('Arial', 'B', 10);
$pdf->SetXY(10, 5);
$pdf->Cell(75, 30, "", 1);
$pdf->Image('img/latkic_fi.png', 15, 8, 40);

// Código SGI segun fecha (comparación inclusiva)
if ($fechaFI >= '2023-11-09') {
  $pdf->SetFont('Arial','',8);
  $pdf->SetXY(10, 25);
  $pdf->Cell(75, 10, toLatin1(" Código SGI: LATKIC-OP-RF00022/Rev.#01/19/02/2025"), 1, 0, 'C');
}

// Encabezado empresa
$pdf->SetFont('Arial','B',12);
$pdf->SetXY(60, 10); $pdf->Cell(0, 1, toLatin1("LATKIC S.A."), 0, 2, 'C');
$pdf->SetFont('Arial','',10);
$pdf->SetXY(60, 15); $pdf->Cell(0, 1, toLatin1("DEPOSITO ADUANERO PÚBLICO"), 0, 2, 'C');
$pdf->SetXY(60, 19); $pdf->Cell(0, 1, toLatin1("RUC: 0992684119001"), 0, 2, 'C');
$pdf->SetFont('Arial','',8);
$pdf->SetXY(60, 23); $pdf->Cell(0, 1, toLatin1("Quito: Km 23 de la Vía Interoceánica - Pifo por la calle de Techint y Carnasa"), 0, 2, 'C');
$pdf->SetXY(60, 26); $pdf->Cell(0, 1, toLatin1("Guayaquil: Vía a la Costa Km 24 S/N Solar 1 Parroquia Chongón"), 0, 2, 'C');

// Título
$pdf->SetFillColor(207,200,198);
$pdf->SetFont('Arial','B',10);
$pdf->SetXY(10, 33); $pdf->Cell(333, 5, toLatin1("FACTURA INFORMATIVA"), 1, 2, 'C', true);

// Bloque datos cortos
$pdf->SetXY(10, 38); $pdf->SetFont('Arial','',7); $pdf->Cell(333, 17, "", 1);
$pdf->SetFont('Arial','B',10);
$pdf->SetXY(10, 38);  $pdf->Cell(75,5, toLatin1('FECHA:'), 1);
$pdf->SetFont('Arial','',10);
$pdf->SetXY(85, 38);  $pdf->Cell(110,5, toLatin1($fechaFI), 1);
$pdf->SetFont('Arial','B',10);
$pdf->SetXY(195,38);  $pdf->Cell(50,5, 'NO.:', 1, 0, 'C');
$pdf->SetFont('Arial','',10);
$pdf->SetXY(245,38);  $pdf->Cell(98,5, toLatin1($numero), 1, 0, 'C');

// Cliente
$pdf->SetFont('Arial','B',10);
$pdf->SetXY(10, 43); $pdf->Cell(75,5, toLatin1('CLIENTE:'), 1);
$pdf->SetFont('Arial','',10);
$pdf->SetXY(85, 43); $pdf->Cell(258,5, toLatin1($nombreCliente), 1);

// Identificación
$pdf->SetFont('Arial','B',10);
$pdf->SetXY(10, 48); $pdf->Cell(75,5, toLatin1('TIPO DE IDENTICACIÓN:'),1);
$pdf->SetFont('Arial','',10);
$pdf->SetXY(85, 48); $pdf->Cell(110,5, toLatin1($tipoId),1);
$pdf->SetFont('Arial','B',10);
$pdf->SetXY(195,48); $pdf->Cell(50,5, toLatin1('N° IDENTIFICACIÓN:'),1,0,'C');
$pdf->SetFont('Arial','',10);
$pdf->SetXY(245,48); $pdf->Cell(98,5, toLatin1($identificacion),1);

// Información general
$pdf->SetFillColor(207,200,198);
$pdf->SetFont('Arial','B',10);
$pdf->SetXY(10,53); $pdf->Cell(333,5, toLatin1('INFORMACIÓN GENERAL'),1,2,'C',true);

$pdf->SetXY(10,58); $pdf->SetFont('Arial','',10); $pdf->Cell(333,15,"",1);
$pdf->SetFont('Arial','B',10);
$pdf->SetXY(10,58);  $pdf->Cell(40,5, 'PEDIDO:',1,0,'C');
$pdf->SetFont('Arial','',10);
$pdf->SetXY(50,58);  $pdf->Cell(145,5, toLatin1($pedido),1,0,'C');

$pdf->SetFont('Arial','B',10);
$pdf->SetXY(10,63);  $pdf->Cell(40,5, 'DAI 70:',1,0,'C');
$pdf->SetFont('Arial','',10);
$pdf->SetXY(50,63);  $pdf->Cell(145,5, toLatin1($dai),1,0,'C');

$pdf->SetFont('Arial','B',10);
$pdf->SetXY(10,68);  $pdf->Cell(40,5, toLatin1('INICIO DE RÉGIMEN:'),1,0,'C');
$pdf->SetFont('Arial','',10);
$pdf->SetXY(50,68);  $pdf->Cell(145,5, toLatin1($fechaReg),1,0,'C');

$pdf->SetFont('Arial','B',10);
$pdf->SetXY(195,58); $pdf->Cell(50,5, toLatin1('SOLICITUD PREVIA:'),1,0,'C');
$pdf->SetFont('Arial','',10);
$pdf->SetXY(245,58); $pdf->Cell(98,5, toLatin1($sp),1,0,'C');

$pdf->SetFont('Arial','B',10);
$pdf->SetXY(195,63); $pdf->Cell(50,5, toLatin1('CON. EMBARQUE:'),1,0,'C');
$pdf->SetFont('Arial','',10);
$pdf->SetXY(245,63); $pdf->Cell(98,5, toLatin1($embarque),1,0,'C');

$pdf->SetFont('Arial','B',10);
$pdf->SetXY(195,68); $pdf->Cell(50,5, toLatin1('MONEDA:'),1,0,'C');
$pdf->SetFont('Arial','',10);
$pdf->SetXY(245,68); $pdf->Cell(44,5, toLatin1($monDetalle),1,0,'C');
$pdf->SetXY(289,68); $pdf->Cell(54,5, toLatin1($monSiglas),1,0,'C');

// Detalle
$pdf->SetFillColor(207,200,198);
$pdf->SetFont('Arial','B',10);
$pdf->SetXY(10,73); $pdf->Cell(333,5, toLatin1('DETALLE'),1,2,'C',true);

// Encabezados
$pdf->SetFont('Arial','B',8);
$pdf->SetXY(10,78);  $pdf->Cell(15,5,'ITEM',1,0,'C');
$pdf->SetXY(25,78);  $pdf->Cell(40,5,'MARCA',1,0,'C');
$pdf->SetXY(65,78);  $pdf->Cell(50,5,toLatin1('DESCRIPCIÓN'),1,0,'C');
$pdf->SetXY(115,78); $pdf->Cell(20,5,'COLOR',1,0,'C');
$pdf->SetXY(135,78); $pdf->Cell(85,5,'MODELO',1,0,'C');
$pdf->SetXY(220,78); $pdf->Cell(15,5,toLatin1('AÑO'),1,0,'C');
$pdf->SetXY(235,78); $pdf->Cell(40,5,'CHASIS',1,0,'C');
$pdf->SetXY(275,78); $pdf->Cell(38,5,'MOTOR',1,0,'C');
$pdf->SetXY(313,78); $pdf->Cell(30,5,'VALOR FOB',1,0,'C');

// Iteración de filas
$y = 83;
$lineasPagina = 17; // después de 17 filas, salto de página
$contLinea = 0;

// Totales (acumular por fila correctamente)
$totalCantidad = 0.0;
$totalFOB      = 0.0;
$totalSeguro   = 0.0;
$totalFlete    = 0.0;

foreach ($items as $row) {
  // Salto de página si es necesario
  if ($contLinea >= $lineasPagina) {
    $pdf->AddPage();
    $y = 10;
    $contLinea = 0;
  }

  $pdf->SetFont('Arial','',9);

  // ITEM
  $pdf->SetXY(10, $y);   $pdf->Cell(15,5, toLatin1((string)$row['item']), 1, 0, 'C');
  // MARCA
  $pdf->SetXY(25, $y);   $pdf->Cell(40,5, toLatin1((string)$row['marca']), 1, 0, 'C');

  // DESCRIPCION (multiline si es larga)
  $desc = (string)$row['descripcion'];
  if (mb_strlen($desc) >= 29) {
    $pdf->SetXY(65, $y);
    $pdf->SetFont('Arial','',9);
    $pdf->MultiCell(50, 2.5, toLatin1($desc), 1, 'J');
  } else {
    $pdf->SetXY(65, $y); $pdf->Cell(50,5, toLatin1($desc), 1, 0, 'C');
  }

  $pdf->SetFont('Arial','',9);
  $pdf->SetXY(115, $y); $pdf->Cell(20,5, toLatin1((string)$row['color']), 1, 0, 'C');

  // MODELO (puede ser largo)
  $modelo = (string)$row['modelo'];
  if (mb_strlen($modelo) >= 45) {
    $pdf->SetXY(135, $y);
    $pdf->SetFont('Arial','',8);
    $pdf->MultiCell(85, 5, toLatin1($modelo), 1, 'J');
  } else {
    $pdf->SetXY(135, $y);
    $pdf->Cell(85,5, toLatin1($modelo), 1, 0, 'C');
  }
  $pdf->SetFont('Arial','',9);

  $pdf->SetXY(220, $y); $pdf->Cell(15,5, toLatin1((string)$row['otras']), 1, 0, 'C'); // Año en 'otras'
  $pdf->SetXY(235, $y); $pdf->Cell(40,5, toLatin1((string)$row['chasis']), 1, 0, 'C');
  $pdf->SetXY(275, $y); $pdf->Cell(38,5, toLatin1((string)$row['motor']), 1, 0, 'C');
  $pdf->SetXY(313, $y); $pdf->Cell(30,5, toLatin1($monSiglas . fmtNum((float)$row['fobitem'])), 1, 0, 'R');

  // Totales por fila
  $totalCantidad += (float)$row['cantidad'];
  $totalFOB      += (float)$row['fobitem'];
  $totalSeguro   += (float)$row['seguro'];
  $totalFlete    += (float)$row['transporte'];

  // Actualizar estado del datadai
  if (!empty($row['id_datadai'])) {
    $updDet = $conn->prepare("UPDATE tb_datadai SET id_estado = 6 WHERE iddatadai = ?");
    $updDet->bind_param('i', $row['id_datadai']);
    $updDet->execute();
  }

  $y += 5;
  $contLinea++;
}

// Pie de totales
$pdf->SetFont('Arial','',8);
$pdf->SetXY(10, $y);
$pdf->Cell(333,5, toLatin1("UNIDADES COMERCIALES: " . (string)$totalCantidad), 1, 0, 'L');

// Cálculo de valor en aduana
// Si fechaFI <= 2023-12-31 usar (FOB + SEGURO + FLETE) según tu comentario de “nueva resolución”
$fechaCorte = '2023-12-31';
$valorAduana = $totalFOB + $totalSeguro + $totalFlete;

// Bloque totales a la derecha
$pdf->SetFont('Arial','B',8);
$pdf->SetXY(275, $y+5);  $pdf->Cell(35,7,'TOTAL FOB:',1,0,'R');
$pdf->SetXY(310, $y+5);  $pdf->Cell(33,7, toLatin1($monSiglas . fmtNum($totalFOB)),1,0,'R');

$pdf->SetXY(275, $y+12); $pdf->Cell(35,7,'FLETE',1,0,'R');
$pdf->SetXY(310, $y+12); $pdf->Cell(33,7, toLatin1($monSiglas . fmtNum($totalFlete)),1,0,'R');

$pdf->SetXY(275, $y+19); $pdf->Cell(35,7,'SEGURO',1,0,'R');
$pdf->SetXY(310, $y+19); $pdf->Cell(33,7, toLatin1($monSiglas . fmtNum($totalSeguro)),1,0,'R');

$pdf->SetXY(275, $y+26); $pdf->Cell(35,7, toLatin1('VALOR EN ADUANA:'),1,0,'R');
$pdf->SetXY(310, $y+26); $pdf->Cell(33,7, toLatin1($monSiglas . fmtNum($valorAduana)),1,0,'R');

// Firma
$pdf->SetXY(10, $y+5);
$pdf->SetFont('Arial','B',8);
$pdf->Cell(265,28,"",1);

$pdf->SetXY(10, $y+14);
$pdf->SetFont('Arial','B',8);
$pdf->Cell(265,20, toLatin1("___________________________________"), 0, 2, 'C');

$pdf->SetXY(10, $y+17);
$pdf->SetFont('Arial','',8);
$pdf->Cell(265,20, toLatin1("FIRMA AUTORIZADA LATKIC"), 0, 2, 'C');

// Salida
$pdf->Output();
