<?php
// views/pages/api_app/upload_foto.php
declare(strict_types=1);

// ====== CORS / Headers ======
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *'); // Restringe si lo necesitas
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

// ====== DEBUG (activar con ?debug=1) ======
$DEBUG = isset($_GET['debug']) && $_GET['debug'] == '1';
if ($DEBUG) {
  ini_set('display_errors','1');
  ini_set('display_startup_errors','1');
  error_reporting(E_ALL);
}

// ====== Respuesta helper ======
function jres(int $code, array $payload) {
  http_response_code($code);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE);
  exit;
}

// ====== (Opcional) Sesión / Auth ======
// session_start();
// if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES') {
//   jres(401, ['success'=>false, 'message'=>'No autorizado']);
// }

// ====== Configuración de rutas ======
// Ruta física base donde se guardarán los archivos.
// Este script está en: views/pages/api_app/upload_foto.php
// Subimos a:         views/pages/uploads/
$BASE_DIR = realpath(__DIR__ . '/..'); // views/pages
if ($BASE_DIR === false) {
  jres(500, ['success'=>false, 'message'=>'No se pudo resolver BASE_DIR']);
}
$BASE_DIR .= '/uploads';               // views/pages/uploads

// URL pública base que apunta a la carpeta anterior:
$BASE_URL = 'https://latkic.com/latkic_2.0/views/pages/uploads';

// Tamaño máximo 10MB
$MAX_SIZE = 10 * 1024 * 1024;

// Extensiones/MIME permitidos
$ALLOWED_EXTS  = ['jpg','jpeg','png','webp'];
$ALLOWED_MIMES = ['image/jpeg','image/png','image/webp'];

// ====== Util ======
function slug($s) {
  $s = (string)$s;
  if ($s === '') return 'file';
  if (function_exists('iconv')) {
    $tmp = @iconv('UTF-8','ASCII//TRANSLIT//IGNORE',$s);
    if ($tmp !== false) $s = $tmp;
  }
  $s = preg_replace('~[^A-Za-z0-9_.-]+~','-', $s);
  $s = trim($s, '-');
  return $s ?: 'file';
}
function detect_mime($tmpPath, $fallbackExt) {
  // 1) fileinfo si está disponible
  if (class_exists('finfo')) {
    $f = new finfo(FILEINFO_MIME_TYPE);
    $m = $f->file($tmpPath);
    if ($m) return $m;
  }
  // 2) getimagesize
  $gi = @getimagesize($tmpPath);
  if ($gi && isset($gi['mime']) && $gi['mime']) return $gi['mime'];
  // 3) mime_content_type (si está)
  if (function_exists('mime_content_type')) {
    $m = @mime_content_type($tmpPath);
    if ($m) return $m;
  }
  // 4) fallback según extensión
  $fallbackExt = strtolower($fallbackExt);
  if (in_array($fallbackExt, ['jpg','jpeg'])) return 'image/jpeg';
  if ($fallbackExt === 'png') return 'image/png';
  if ($fallbackExt === 'webp') return 'image/webp';
  return 'application/octet-stream';
}

// ====== Validaciones ======
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  jres(405, ['success'=>false,'message'=>'Método no permitido']);
}
if (empty($_FILES['foto']) || !is_uploaded_file($_FILES['foto']['tmp_name'])) {
  jres(400, ['success'=>false,'message'=>'Falta el archivo "foto" (multipart/form-data)']);
}

// parámetro modulo (carpeta lógica)
$modulo = isset($_GET['modulo']) ? trim((string)$_GET['modulo']) : '';
if ($modulo === '') $modulo = 'general';
$modulo = slug($modulo);

// tamaño
$size = (int)($_FILES['foto']['size'] ?? 0);
if ($size <= 0) jres(400, ['success'=>false,'message'=>'Archivo vacío']);
if ($size > $MAX_SIZE) jres(413, ['success'=>false,'message'=>'Archivo demasiado grande (máx 10 MB)']);

// nombre y ext
$origName = $_FILES['foto']['name'] ?? 'foto';
$ext = strtolower(pathinfo($origName, PATHINFO_EXTENSION));
if (!in_array($ext, $ALLOWED_EXTS, true)) {
  jres(400, ['success'=>false,'message'=>'Extensión no permitida. Use: '.implode(', ', $ALLOWED_EXTS)]);
}

// mime detectado
$tmpPath = $_FILES['foto']['tmp_name'];
$mimeDetected = detect_mime($tmpPath, $ext);
if (!in_array($mimeDetected, $ALLOWED_MIMES, true)) {
  jres(400, ['success'=>false,'message'=>"Tipo MIME no permitido: $mimeDetected"]);
}

// ====== Verificar/crear carpetas ======
$y = date('Y'); $m = date('m');
$destDir = $BASE_DIR . "/{$modulo}/{$y}/{$m}";

// Verificar que el padre es escribible
$parent = dirname($destDir);
if (!is_dir($parent)) {
  if (!mkdir($parent, 0775, true)) {
    jres(500, ['success'=>false,'message'=>'No se pudo crear carpeta padre', 'debug'=>$DEBUG?['parent'=>$parent]:null]);
  }
}
if (!is_dir($destDir)) {
  if (!mkdir($destDir, 0775, true)) {
    jres(500, ['success'=>false,'message'=>'No se pudo crear carpeta destino', 'debug'=>$DEBUG?['destDir'=>$destDir]:null]);
  }
}

// Comprobar permisos de escritura
if (!is_writable($destDir)) {
  jres(500, ['success'=>false,'message'=>'Directorio destino sin permisos de escritura', 'debug'=>$DEBUG?['destDir'=>$destDir]:null]);
}

// ====== Nombre final ======
$baseSlug = slug(pathinfo($origName, PATHINFO_FILENAME));
$rand     = bin2hex(random_bytes(4));
$fname    = "{$baseSlug}_" . date('YmdHis') . "_{$rand}.{$ext}";
$destPath = $destDir . '/' . $fname;

// ====== Mover ======
if (!@move_uploaded_file($tmpPath, $destPath)) {
  // Diagnóstico extra en DEBUG
  $err = error_get_last();
  jres(500, [
    'success'=>false,
    'message'=>'No se pudo guardar el archivo',
    'debug'=>$DEBUG ? [
      'destPath'=>$destPath,
      'tmpPath'=>$tmpPath,
      'error_last'=>$err
    ] : null
  ]);
}

// endurecer permisos
@chmod($destPath, 0644);

// ====== Respuesta ======
$relativePath = "{$modulo}/{$y}/{$m}/{$fname}";
$fileUrl      = rtrim($BASE_URL, '/') . '/' . $relativePath;

jres(200, [
  'success'       => true,
  'message'       => 'OK',
  'filename'      => $fname,
  'size'          => $size,
  'mime'          => $mimeDetected,
  'relative_path' => $relativePath, // <- ruta para BD
  'url'           => $fileUrl       // <- URL pública
]);
