<?php
// views/pages/api_app/andenes_disponibles.php
// — API: Lista andenes disponibles (libres) => activo <> 1 (o NULL) y condicion = 1 (por defecto)
//
// GET params opcionales:
//   id_ubicacion (int)  -> filtra por ubicación exacta
//   q (string)          -> búsqueda por etiqueta (LIKE %q%)
//   limit (int)         -> tamaño de página (default 100)
//   offset (int)        -> desplazamiento (default 0)
//   incluir_todos (0/1) -> si =1 ignora condicion=1 (trae todos los ANDENES libres sin filtrar por condicion)
//
// Ejemplos:
//   andenes_disponibles.php
//   andenes_disponibles.php?id_ubicacion=3
//   andenes_disponibles.php?q=PUERTA&limit=50&offset=0

declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

date_default_timezone_set('America/Guayaquil');

$respond = function (int $code, array $payload): void {
  http_response_code($code);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
};

// Errores a JSON
set_exception_handler(function (Throwable $e) use ($respond) {
  $respond(500, ['success' => false, 'error' => 'EXCEPTION', 'message' => $e->getMessage(), 'timestamp' => date('c')]);
});
set_error_handler(function ($errno, $errstr, $errfile, $errline) {
  throw new ErrorException($errstr, 0, $errno, $errfile, $errline);
});

require_once __DIR__ . '/../conex.php'; // ajusta ruta si aplica
if (!isset($conn) || !($conn instanceof mysqli)) {
  $respond(500, ['success' => false, 'error' => 'DB_CONNECTION', 'message' => 'Conexión mysqli inválida.']);
}

mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
$conn->set_charset('utf8mb4');

// Parámetros
$idUbicacion   = isset($_GET['id_ubicacion']) ? (int)$_GET['id_ubicacion'] : 0;
$q             = isset($_GET['q']) ? trim((string)$_GET['q']) : '';
$limit         = isset($_GET['limit'])  ? max(1, (int)$_GET['limit'])  : 100;
$offset        = isset($_GET['offset']) ? max(0, (int)$_GET['offset']) : 0;
$incluirTodos  = isset($_GET['incluir_todos']) ? ((int)$_GET['incluir_todos'] === 1) : false;

// Base: libres = activo <> 1 OR activo IS NULL
$where  = 'WHERE (a.activo <> 1 OR a.activo IS NULL)';
$params = [];
$types  = '';

// Por defecto solo andenes habilitados (condicion = 1); si incluir_todos=1, se ignora este filtro
if (!$incluirTodos) {
  $where .= ' AND a.condicion = 1';
}

// Filtro por ubicación
if ($idUbicacion > 0) {
  $where .= ' AND a.id_ubicacion = ?';
  $params[] = $idUbicacion;
  $types   .= 'i';
}

// Búsqueda por etiqueta
if ($q !== '') {
  $where .= ' AND a.etiqueta LIKE ?';
  $params[] = '%' . $q . '%';
  $types   .= 's';
}

// COUNT
$sqlCount = "
  SELECT COUNT(*) AS total
  FROM tb_anden AS a
  $where
";

// DATA
$sqlData = "
  SELECT
    a.id,
    a.id_ubicacion,
    a.etiqueta,
    a.activo,
    a.condicion
  FROM tb_anden AS a
  $where
  ORDER BY a.id_ubicacion ASC, a.etiqueta ASC, a.id ASC
  LIMIT ? OFFSET ?
";

// Ejecutar COUNT
$stmtC = $conn->prepare($sqlCount);
if ($types !== '') { $stmtC->bind_param($types, ...$params); }
$stmtC->execute();
$resC  = $stmtC->get_result();
$rowC  = $resC->fetch_assoc();
$total = (int)($rowC['total'] ?? 0);
$stmtC->close();

// Ejecutar DATA (agregar limit/offset a los params)
$stmtD = $conn->prepare($sqlData);
$typesData   = $types . 'ii';
$paramsData  = $params;
$paramsData[] = $limit;
$paramsData[] = $offset;

// bind_param con referencias
$tmp = [];
$tmp[] = $typesData;
foreach ($paramsData as $k => $v) { $tmp[] = &$paramsData[$k]; }
call_user_func_array([$stmtD, 'bind_param'], $tmp);

$stmtD->execute();
$resD = $stmtD->get_result();

$data = [];
while ($r = $resD->fetch_assoc()) {
  $data[] = [
    'id'           => (int)$r['id'],
    'id_ubicacion' => isset($r['id_ubicacion']) ? (int)$r['id_ubicacion'] : null,
    'etiqueta'     => $r['etiqueta'],
    'activo'       => is_null($r['activo']) ? null : (int)$r['activo'],
    'condicion'    => is_null($r['condicion']) ? null : (int)$r['condicion'],
  ];
}
$stmtD->close();

$respond(200, [
  'success'   => true,
  'timestamp' => date('c'),
  'filters'   => [
    'id_ubicacion'  => $idUbicacion,
    'q'             => $q,
    'incluir_todos' => $incluirTodos ? 1 : 0,
    'limit'         => $limit,
    'offset'        => $offset,
  ],
  'total'     => $total,
  'count'     => count($data),
  'data'      => $data,
]);
