<?php
// api/nacionalizado_list.php — Lista para DataTable de Nacionalizados (BS5)
declare(strict_types=1);
session_start();

/* ====== Autorización ====== */
if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES') {
  http_response_code(401);
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode(['data' => [], 'message' => 'No autorizado'], JSON_UNESCAPED_UNICODE);
  exit;
}

$id_usuario = (int)($_SESSION['idusuario'] ?? 0);
$permiso    = $_SESSION['permiso'] ?? '';

header('Content-Type: application/json; charset=utf-8');

/* ====== Conexión ====== */
require_once __DIR__ . '/../conex.php'; // corrige la ruta
$db = $conn ?? ($conex ?? null);

if (!$db instanceof mysqli) {
  http_response_code(500);
  echo json_encode(['data' => [], 'message' => 'Sin conexión a BD'], JSON_UNESCAPED_UNICODE);
  exit;
}
mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'");

/*
  Criterio “nacionalizado” según tu lógica:
  d.id_estado = 8, d.dai = 10, d.condicion = 1
  y filtra por bodega del usuario en sesión (d.id_bodega = us.id_bodega)
*/
$sql = "
  SELECT DISTINCT
         d.iddatadai,
         d.dai,               -- normalmente 10
         ds.numeros,          -- Nº DAI 10
         d.item,
         d.marca,
         d.modelo,
         d.color,
         d.chasis,
         d.motor,
         d.id_guia2,
         ds.iddaisalida
  FROM tb_datadai AS d
  INNER JOIN tb_daisalida AS ds
          ON ds.iddaisalida = d.id_daisalida
  LEFT  JOIN tb_bodega AS b
          ON b.idbodega = ds.id_bodega
  INNER JOIN tb_usuario AS us
          ON us.idusuario = ?
         AND d.id_bodega  = us.id_bodega
  WHERE d.id_estado = 8
    AND d.dai = 10
    AND d.condicion = 1
  ORDER BY ds.numeros, d.item
";

$stmt = mysqli_prepare($db, $sql);
if (!$stmt) {
  http_response_code(500);
  echo json_encode(['data'=>[], 'message'=>'Error preparando SQL', 'error'=>mysqli_error($db)], JSON_UNESCAPED_UNICODE);
  exit;
}
mysqli_stmt_bind_param($stmt, 'i', $id_usuario);
mysqli_stmt_execute($stmt);
$res = mysqli_stmt_get_result($stmt);

$data = [];
while ($row = mysqli_fetch_assoc($res)) {
  $iddatadai   = (int)($row['iddatadai']   ?? 0);
  $dai         = (string)($row['dai']       ?? '');
  $numeros     = (string)($row['numeros']   ?? '');
  $item        = (string)($row['item']      ?? '');
  $marca       = (string)($row['marca']     ?? '');
  $modelo      = (string)($row['modelo']    ?? '');
  $color       = (string)($row['color']     ?? '');
  $chasis      = (string)($row['chasis']    ?? '');
  $motor       = (string)($row['motor']     ?? '');
  $id_guia2    = $row['id_guia2']; // puede ser NULL
  $iddaisalida = (int)($row['iddaisalida'] ?? 0);

  // Botones (BS5 + Font Awesome 6)
	$verpdf = '<a href="../../pdf/index.php?itemdai=' . $iddatadai . '" '
	        . 'class="btn btn-primary" '
	        . 'title="Tarja de Ingreso" data-bs-toggle="tooltip">'
	        . '<i class="fa-solid fa-file-lines" aria-hidden="true"></i></a>';

	$editar = '<a id="btnnotifi' . $iddatadai . '" '
	        . 'data-ideq="' . $iddatadai . '" '
	        . 'data-ideq2="' . htmlspecialchars($numeros, ENT_QUOTES, 'UTF-8') . '" '
	        . 'data-ideq3="' . $iddaisalida . '" '
	        . 'class="btn btn-warning" '
	        . 'title="Editar" data-bs-toggle="modal" data-bs-target="#editar_valores">'
	        . '<i class="fa-solid fa-pen-to-square" aria-hidden="true"></i></a>';

	// Si también usas el botón "Despachar" en algún lado:
	$salida = '<a href="actionFinalizar.php?idsp=' . $iddatadai . '" '
	        . 'onclick="return confirm(\'Confirmar la salida?\')" '
	        . 'class="btn btn-success" title="Despachar" data-bs-toggle="tooltip">'
	        . '<i class="fa-solid fa-right-from-bracket" aria-hidden="true"></i></a>';


	  $accion = $verpdf . ' ' . $editar;

  if (!is_null($id_guia2)) {
    // Cambia el color y agrega botón de despachar si ya tiene guía
    $verpdf = '<a href="../../pdf/index.php?itemdai=' . $iddatadai . '" '
            . 'class="btn btn-info" '
            . 'title="Tarja de salida" data-bs-toggle="tooltip">'
            . '<i class="fa fa-file-o" aria-hidden="true"></i></a>';

    $salida = '<a href="actionFinalizar.php?idsp=' . $iddatadai . '" '
            . 'onclick="return confirm(\'Confirmar la salida?\')" '
            . 'class="btn btn-success" title="Despachar" data-bs-toggle="tooltip">'
            . '<i class="fa fa-sign-out" aria-hidden="true"></i></a>';

    $accion = $verpdf . ' ' . $editar . ' ' . $salida;
  }

  $data[] = [
    'regimen' => $dai,
    'dai10'   => $numeros,
    'item'    => $item,
    'marca'   => $marca,
    'modelo'  => $modelo,
    'color'   => $color,
    'chasis'  => $chasis,
    'motor'   => $motor,
    // Este campo es opcional para tu DataTable; si no lo usas, simplemente se ignora
    'accion'  => $accion,
  ];
}

echo json_encode(['data' => $data], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
