<?php
// Reemplaza el set de talleres para un incidente (compatible PHP 7+)
declare(strict_types=1);
require_once __DIR__ . '/_api_bootstrap.php';

try {
  // Leer JSON
  $inRaw = file_get_contents('php://input');
  $in = json_decode($inRaw, true);
  if (!is_array($in)) { $in = []; }

  $incident_id = isset($in['incident_id']) ? (int)$in['incident_id'] : 0;

  // Normalizar talleres_ids (enteros únicos > 0) sin arrow functions
  $talleres_ids = [];
  if (isset($in['talleres_ids']) && is_array($in['talleres_ids'])) {
    foreach ($in['talleres_ids'] as $v) {
      $iv = (int)$v;
      if ($iv > 0) { $talleres_ids[] = $iv; }
    }
    $talleres_ids = array_values(array_unique($talleres_ids));
  }

  if ($incident_id <= 0) {
    api_json(['success'=>false,'message'=>'incident_id requerido'], 422);
  }

  // Asegurar tabla
  $ddl = "CREATE TABLE IF NOT EXISTS tb_incidente_taller (
            incident_id INT NOT NULL,
            id_taller   INT NOT NULL,
            updated_at  TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (incident_id, id_taller),
            KEY idx_taller (id_taller)
          ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
  if (!$db->query($ddl)) {
    api_json(['success'=>false,'message'=>'Error creando tabla','sql_error'=>$db->error], 500);
  }

  // (Opcional) Validar existencia/activos en tb_taller
  if (!empty($talleres_ids)) {
    $marks = implode(',', array_fill(0, count($talleres_ids), '?'));
    $sqlV  = "SELECT idtaller FROM tb_taller WHERE condicion=1 AND idtaller IN ($marks)";
    $stmtV = $db->prepare($sqlV);
    if (!$stmtV) {
      api_json(['success'=>false,'message'=>'Error en prepare(validación)','sql_error'=>$db->error], 500);
    }
    $types = str_repeat('i', count($talleres_ids));
    // @phpstan-ignore-next-line
    $stmtV->bind_param($types, ...$talleres_ids);
    if (!$stmtV->execute()) {
      api_json(['success'=>false,'message'=>'Error ejecutando validación','sql_error'=>$stmtV->error], 500);
    }
    $rv = $stmtV->get_result();
    $validos = [];
    while ($x = $rv->fetch_assoc()) { $validos[] = (int)$x['idtaller']; }
    $missing = array_values(array_diff($talleres_ids, $validos));
    if (!empty($missing)) {
      api_json([
        'success'=>false,
        'message'=>'Taller(es) inexistente(s) o inactivo(s)',
        'detalle'=>['no_encontrados'=>$missing]
      ], 422);
    }
  }

  // Transacción
  if (!$db->begin_transaction()) {
    api_json(['success'=>false,'message'=>'No se pudo iniciar transacción','sql_error'=>$db->error], 500);
  }

  // Limpiar anteriores
  $stmtDel = $db->prepare("DELETE FROM tb_incidente_taller WHERE incident_id=?");
  if (!$stmtDel) {
    $db->rollback();
    api_json(['success'=>false,'message'=>'Error en prepare(DELETE)','sql_error'=>$db->error], 500);
  }
  $stmtDel->bind_param('i', $incident_id);
  if (!$stmtDel->execute()) {
    $db->rollback();
    api_json(['success'=>false,'message'=>'Error ejecutando DELETE','sql_error'=>$stmtDel->error], 500);
  }

  // Insertar nuevos
  if (!empty($talleres_ids)) {
    $stmtIns = $db->prepare("INSERT INTO tb_incidente_taller (incident_id, id_taller) VALUES (?, ?)");
    if (!$stmtIns) {
      $db->rollback();
      api_json(['success'=>false,'message'=>'Error en prepare(INSERT)','sql_error'=>$db->error], 500);
    }
    foreach ($talleres_ids as $tid) {
      $stmtIns->bind_param('ii', $incident_id, $tid);
      if (!$stmtIns->execute()) {
        $db->rollback();
        api_json([
          'success'=>false,
          'message'=>'Error insertando taller',
          'sql_error'=>$stmtIns->error,
          'detalle'=>['id_taller'=>$tid]
        ], 500);
      }
    }
  }

  // Commit
  if (!$db->commit()) {
    $db->rollback();
    api_json(['success'=>false,'message'=>'Error haciendo commit','sql_error'=>$db->error], 500);
  }

  api_json(['success'=>true]);

} catch (Throwable $e) {
  if (isset($db) && ($db->errno ?? 0)) { @mysqli_rollback($db); }
  api_json(['success'=>false,'message'=>'Error guardando talleres','detail'=>$e->getMessage()], 500);
}
