<?php
// views/pages/api/facturas_crear.php
// Crear cabecera de Factura Informativa:
// - Requiere cliente (id_cliente) o exonerado=1 + nombre
// - Requiere al menos un iddatadai (iddatadai_list) para obtener SIGLA (tb_bodega) y armar numero SIGLA+AÑO-####
// - Usa fecha del servidor (Y-m-d)
// - id_moneda = NULL (no se usa)
// - Opcional: recibe cantidad/totalfob/totalflete/totalseguro/total y los guarda tal cual

declare(strict_types=1);
session_start();

/* ===== Respuesta JSON & CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

/* ===== Preflight ===== */
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Método ===== */
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Método no permitido']);
  exit;
}

/* ===== Robustez: capturar fatales como JSON ===== */
error_reporting(E_ALL);
ini_set('display_errors', '0');
ob_start();
register_shutdown_function(function () {
  $e = error_get_last();
  if ($e && in_array($e['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
    if (ob_get_level()) { ob_end_clean(); }
    http_response_code(500);
    echo json_encode([
      'success'=>false,
      'message'=>'Error fatal en servidor',
      'detail'=>($e['message'] ?? '') . ' @ ' . ($e['file'] ?? '') . ':' . ($e['line'] ?? 0)
    ]);
  }
});

/* ===== Conexión ===== */
require_once __DIR__ . '/../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db) {
  if (ob_get_level()) ob_end_clean();
  echo json_encode(['success'=>false,'message'=>'Sin conexión a la base de datos']);
  exit;
}
mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'"); // Ecuador

/* ===== Entradas ===== */
$id_cliente   = (int)($_POST['id_cliente'] ?? 0);
$exonerado    = (int)($_POST['exonerado'] ?? 0);
$nombre       = trim((string)($_POST['nombre'] ?? ''));
$cedula       = trim((string)($_POST['ruc'] ?? ''));          // input del modal

// Totales opcionales (si no llegan, 0)
$cantidad     = (int)($_POST['cantidad']      ?? 0);
$totalfob     = (float)($_POST['totalfob']    ?? 0);
$totalflete   = (float)($_POST['totalflete']  ?? 0);
$totalseguro  = (float)($_POST['totalseguro'] ?? 0);
$total        = (float)($_POST['total']       ?? 0);

// Lista de iddatadai (JSON string desde FormData)
$idd_list_raw = (string)($_POST['iddatadai_list'] ?? '');
$idd_list     = [];
if ($idd_list_raw !== '') {
  $tmp = json_decode($idd_list_raw, true);
  if (is_array($tmp)) {
    foreach ($tmp as $v) {
      $n = (int)$v;
      if ($n > 0) $idd_list[] = $n;
    }
  }
}

/* ===== Validaciones ===== */
if ($id_cliente <= 0 && !($exonerado === 1 && $nombre !== '')) {
  if (ob_get_level()) ob_end_clean();
  echo json_encode(['success'=>false,'message'=>'Debe especificar cliente o nombre (exonerado=1).']);
  exit;
}
if (empty($idd_list)) {
  if (ob_get_level()) ob_end_clean();
  echo json_encode(['success'=>false,'message'=>'Se requiere al menos un iddatadai para determinar la bodega (SIGLA).']);
  exit;
}

/* ===== Determinar SIGLA y relaciones a partir de un iddatadai =====
   Preferimos el primero de la lista. Se asume:
   - tb_datadai: iddatadai, id_bodega, id_daientrada, id_solicitudprevia (nombres según tu BD)
   - tb_bodega:  idbodega, sigla
*/
$iddatadai_ref = (int)$idd_list[0];

$sqlInfo = "SELECT 
              d.id_daientrada,
              d.id_solicitudprevia,   -- ajusta si tu columna es id_solicitud_previa o similar
              d.id_bodega,
              b.sigla
            FROM tb_datadai AS d
            LEFT JOIN tb_bodega AS b ON b.idbodega = d.id_bodega
            WHERE d.iddatadai = ?";
$stmtInfo = mysqli_prepare($db, $sqlInfo);
mysqli_stmt_bind_param($stmtInfo, 'i', $iddatadai_ref);
mysqli_stmt_execute($stmtInfo);
$rsInfo = mysqli_stmt_get_result($stmtInfo);

$id_daientrada       = 0;
$id_solicitud_previa = 0;
$sigla               = '';
if ($row = mysqli_fetch_assoc($rsInfo)) {
  $id_daientrada       = (int)($row['id_daientrada'] ?? 0);
  // El nombre exacto de la columna puede variar: id_solicitudprevia vs id_solicitud_previa
  $id_solicitud_previa = (int)($row['id_solicitudprevia'] ?? ($row['id_solicitud_previa'] ?? 0));
  $sigla               = trim((string)($row['sigla'] ?? ''));
}
mysqli_free_result($rsInfo);
mysqli_stmt_close($stmtInfo);

if ($sigla === '') {
  if (ob_get_level()) ob_end_clean();
  echo json_encode(['success'=>false,'message'=>'No se pudo determinar la SIGLA (bodega) desde el DAI proporcionado.']);
  exit;
}

/* ===== Generar número: SIGLA + AÑO + '-' + 4 dígitos ===== */
$anio   = date('Y');
$prefix = $sigla . $anio . '-';

// Tomar el máximo de los 4 últimos dígitos entre los números con ese prefijo
$sqlSeq = "SELECT MAX(CAST(RIGHT(numero, 4) AS UNSIGNED)) AS maxseq
           FROM tb_facturainfo
           WHERE numero LIKE CONCAT(?, '%')";
$stmtSeq = mysqli_prepare($db, $sqlSeq);
mysqli_stmt_bind_param($stmtSeq, 's', $prefix);
mysqli_stmt_execute($stmtSeq);
$rsSeq = mysqli_stmt_get_result($stmtSeq);
$maxseq = 0;
if ($r = mysqli_fetch_assoc($rsSeq)) {
  $maxseq = (int)($r['maxseq'] ?? 0);
}
mysqli_free_result($rsSeq);
mysqli_stmt_close($stmtSeq);

$next   = $maxseq + 1;
$dig4   = str_pad((string)$next, 4, '0', STR_PAD_LEFT);
$numero = $prefix . $dig4;

/* ===== Datos fijos ===== */
$fecha     = date('Y-m-d');
$id_estado = 5;   // estado inicial
$condicion = 1;   // activo

/* ===== INSERT cabecera =====
   Estructura exacta de columnas (según lo que nos diste):
   idfacturainfo,
   numero,
   fecha,
   id_daientrada,
   id_solicitud_previa,
   id_cliente,
   cantidad,
   totalfob,
   totalflete,
   totalseguro,
   total,
   id_moneda,
   exonerado,
   cedula,
   nombre,
   id_estado,
   condicion
*/
$sqlIns = "INSERT INTO tb_facturainfo
            (numero, fecha, id_daientrada, id_solicitud_previa, id_cliente,
             cantidad, totalfob, totalflete, totalseguro, total, id_moneda,
             exonerado, cedula, nombre, id_estado, condicion)
          VALUES
            (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NULL, ?, ?, ?, ?, ?)";

$stmtIns = mysqli_prepare($db, $sqlIns);
if (!$stmtIns) {
  if (ob_get_level()) ob_end_clean();
  echo json_encode(['success'=>false,'message'=>'Error al preparar INSERT (verifica nombres de columnas).']);
  exit;
}

mysqli_stmt_bind_param(
  $stmtIns,
  'ssiiiiddddissii',
  $numero,                // s
  $fecha,                 // s
  $id_daientrada,         // i
  $id_solicitud_previa,   // i
  $id_cliente,            // i
  $cantidad,              // i
  $totalfob,              // d
  $totalflete,            // d
  $totalseguro,           // d
  $total,                 // d
  $exonerado,             // i
  $cedula,                // s
  $nombre,                // s
  $id_estado,             // i
  $condicion              // i
);

$ok = mysqli_stmt_execute($stmtIns);
if (!$ok) {
  $err = mysqli_error($db);
  mysqli_stmt_close($stmtIns);
  if (ob_get_level()) ob_end_clean();
  echo json_encode(['success'=>false,'message'=>'Error al crear factura: '.$err]);
  exit;
}

$idfacturainfo = (int)mysqli_insert_id($db);
mysqli_stmt_close($stmtIns);

/* ===== Éxito ===== */
if (ob_get_level()) ob_end_clean();
echo json_encode([
  'success'        => true,
  'idfacturainfo'  => $idfacturainfo,
  'numero'         => $numero,
  'fecha'          => $fecha,
  'sigla'          => $sigla,
  'anio'           => $anio
]);
