<?php
// api/accesorizacion_leer_multi.php
// Entrada: JSON { "chasis": ["ABC123","XYZ456", ...] }
// Salida: { success: true, data: { total:N, counts:{id_acc:int}, by_chasis:{ "ABC123":[id_acc,...], ... } } }

declare(strict_types=1);
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');

require_once __DIR__ . '/../conex.php';

try {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new RuntimeException('Método no permitido');
    }

    $raw = file_get_contents('php://input');
    $in  = json_decode($raw, true);
    $chasisList = array_values(array_filter(array_map('trim', (array)($in['chasis'] ?? []))));

    if (!$chasisList) {
        echo json_encode(['success'=>true, 'data'=>['total'=>0, 'counts'=>new stdClass(), 'by_chasis'=>new stdClass()]]);
        exit;
    }

    if (!isset($conn) || !($conn instanceof mysqli)) throw new RuntimeException('Sin conexión BD');
    mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
    mysqli_set_charset($conn, 'utf8mb4');

    // 1) Mapear chasis -> iddatadai
    $placeholders = implode(',', array_fill(0, count($chasisList), '?'));
    $sql1 = "SELECT iddatadai, chasis FROM tb_datadai WHERE chasis IN ($placeholders)";
    $stmt1 = $conn->prepare($sql1);
    $types = str_repeat('s', count($chasisList));
    $stmt1->bind_param($types, ...$chasisList);
    $stmt1->execute();
    $res1 = $stmt1->get_result();

    $ids = [];              // iddatadai[]
    $byChasis = [];         // chasis => []
    while ($row = $res1->fetch_assoc()) {
        $ids[] = (int)$row['iddatadai'];
        $byChasis[$row['chasis']] = [];
    }
    $stmt1->close();

    if (!$ids) {
        echo json_encode(['success'=>true, 'data'=>['total'=>0, 'counts'=>new stdClass(), 'by_chasis'=>new stdClass()]]);
        exit;
    }

    // 2) Traer accesorios activos por id_datadai
    $ph2 = implode(',', array_fill(0, count($ids), '?'));
    $sql2 = "
        SELECT a.id_datadai, a.id_accesorios
        FROM tb_accesorizacion a
        WHERE a.estado_acces = 1
          AND a.id_datadai IN ($ph2)
    ";
    $stmt2 = $conn->prepare($sql2);
    $types2 = str_repeat('i', count($ids));
    $stmt2->bind_param($types2, ...$ids);
    $stmt2->execute();
    $res2 = $stmt2->get_result();

    // Para mapear id_datadai -> chasis rápidamente
    // Traemos el mapa al vuelo otra vez o construimos de la consulta 1
    $idToChasis = [];
    // Rehacer consulta 1 rápidamente (o conserva resultados)
    foreach ($byChasis as $ch => $_) {
        // Buscamos su id en tb_datadai, mejor lo traemos de nuevo con un mapa
        // Para evitar otra query, armemos un mapa con la respuesta anterior:
        // Ya no tenemos id->chasis guardado, así que recuperemos desde DB con IN (opcional).
        // Simplificación: guardemos id->chasis durante el paso 1:
    }
    // Rehacer el paso 1 con mapa id->chasis:
    // (Si prefieres evitar, puedes hacer SELECT iddatadai, chasis ... y llenar ambos mapas allí mismo.)
    // Ajuste: volvemos a ejecutar una consulta mínima para tener el mapa (opcional)
    // Pero es más limpio: volvamos a crear el mapa de una forma simple:
    $stmt1 = $conn->prepare("SELECT iddatadai, chasis FROM tb_datadai WHERE iddatadai IN ($ph2)");
    $stmt1->bind_param($types2, ...$ids);
    $stmt1->execute();
    $resMap = $stmt1->get_result();
    while ($r = $resMap->fetch_assoc()) {
        $idToChasis[(int)$r['iddatadai']] = $r['chasis'];
    }
    $stmt1->close();

    $counts = []; // id_acc => cuántos chasis lo tienen
    while ($row = $res2->fetch_assoc()) {
        $idDat = (int)$row['id_datadai'];
        $idAcc = (string)$row['id_accesorios'];
        $ch    = $idToChasis[$idDat] ?? null;
        if ($ch === null) continue;

        if (!isset($byChasis[$ch])) $byChasis[$ch] = [];
        if (!in_array($idAcc, $byChasis[$ch], true)) {
            $byChasis[$ch][] = $idAcc;
        }
        $counts[$idAcc] = ($counts[$idAcc] ?? 0) + 1;
    }
    $stmt2->close();

    // Ordenar arrays por estética (opcional)
    foreach ($byChasis as $ch => $arr) {
        sort($arr, SORT_NATURAL);
        $byChasis[$ch] = $arr;
    }

    echo json_encode([
        'success' => true,
        'data' => [
            'total'      => count($byChasis),
            'counts'     => (object)$counts,      // { "12": 3, "25": 1, ... }
            'by_chasis'  => (object)$byChasis     // { "ABC123":["12","25"], ... }
        ]
    ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['success'=>false, 'error'=>$e->getMessage()]);
}
