<?php 
// alistamiento_pdi.php — SGO · Requerimientos de Alistamiento (PDI) — usa tb_pdi como único almacenamiento
declare(strict_types=1);
session_start();
if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES') {
  header("Location: ../../views/index.php");
  exit();
}
$usuario = $_SESSION['usuario'] ?? '';
$permiso = (int)($_SESSION['permiso'] ?? 0);

require_once __DIR__ . '/conex.php';

date_default_timezone_set('America/Guayaquil');
$hoyFechaHora  = date('Y-m-d\TH:i');               // para <input type="datetime-local">
$manianaFechaH = date('Y-m-d\TH:i', strtotime('+1 day'));
?>
<?php include 'partials/header.php'; ?>
<body>
  <div class="container-fluid">
    <div class="row">
      <?php include 'partials/menu.php'; ?>

      <main class="col-md-9 ms-sm-auto col-lg-10 px-2">
        <?php include 'partials/topbar.php'; ?>

        <div class="d-flex justify-content-between align-items-center mb-2">
          <h6 class="mb-0">Requerimientos de Alistamiento</h6>
          <button type="button" class="btn btn-success btn-sm" data-bs-toggle="modal" data-bs-target="#agregar_factura">
            <i class="fa fa-plus me-1"></i> Agregar Requerimiento
          </button>
        </div>

        <div class="card shadow-sm">
          <div class="card-body p-2">
            <table id="pdi" class="table table-sm table-hover w-100">
              <thead class="table-light">
                <tr>
                  <th>Marca</th>
                  <th>Modelo</th>
                  <th>Color</th>
                  <th>Chasis</th>
                  <th>Lugar</th>
                  <th>Tipo de alistamiento</th>
                  <th>Fecha de requerimiento</th>
                  <th>Fecha aprox. entrega</th>
                  <th>Prioridad</th>
                  <th>Tipo de lavada</th><!-- Select editable -->
                  <th>Facturado a</th>
                </tr>
              </thead>
              <tbody></tbody>
            </table>
          </div>
        </div>
      </main>
    </div>
  </div>

  <!-- Modal: Agregar Requerimiento -->
  <div class="modal fade" id="agregar_factura" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg modal-dialog-centered">
      <div class="modal-content position-relative">
        <!-- Overlay de guardado -->
        <div id="savingOverlay" class="modal-saving-overlay">
          <div class="spinner-border me-2" role="status" aria-hidden="true"></div>
          <strong>Guardando…</strong>
        </div>

        <div class="modal-header py-2">
          <h6 class="modal-title"><i class="fa fa-clipboard-list me-2"></i>Agregar vehículos al requerimiento</h6>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
        </div>
        <div class="modal-body">
          <!-- Cabecera -->
          <div class="row g-2">
            <div class="col-md-4">
              <label class="form-label">TIPO DE ALISTAMIENTO *</label>
              <select class="form-select form-select-sm" id="alistamiento" required>
                <option value="">Seleccione...</option>
                <option value="Con PDI">Con PDI</option>
                <option value="Sin PDI">Sin PDI</option>
              </select>
            </div>

            <div class="col-md-3">
              <label class="form-label">REQUERIDO POR *</label>
              <input type="text" class="form-control form-control-sm" id="requerido" name="requerido"
                     value="<?= htmlspecialchars($usuario, ENT_QUOTES, 'UTF-8') ?>" required readonly>
            </div>

            <div class="col-md-5">
              <label class="form-label" for="lugar">Lugar *</label>
              <select class="form-select form-select-sm" id="lugar" name="lugar" required>
                <option value="">Seleccione…</option>
              </select>
            </div>

            <div class="col-md-4">
              <label class="form-label">FECHA DE REQUERIMIENTO *</label>
              <input class="form-control form-control-sm" type="datetime-local" id="f_requerido"
                     required step="60" value="<?= $hoyFechaHora ?>" min="<?= $hoyFechaHora ?>">
            </div>

            <div class="col-md-4">
              <label class="form-label">FECHA MÁXIMA DE ENTREGA *</label>
              <input class="form-control form-control-sm" type="datetime-local" id="f_entrega"
                     required step="60" value="<?= $manianaFechaH ?>" min="<?= $hoyFechaHora ?>">
            </div>

            <div class="col-md-4">
              <label class="form-label d-block mb-1">Prioridad *</label>
              <div class="btn-group w-100" role="group" aria-label="Prioridad">
                <input type="radio" class="btn-check" name="prioridad" id="prioridad-normal" value="NORMAL"
                       autocomplete="off" <?= (($_POST['prioridad'] ?? 'NORMAL') === 'NORMAL') ? 'checked' : ''; ?> required>
                <label class="btn btn-outline-secondary btn-sm" for="prioridad-normal">NORMAL</label>

                <input type="radio" class="btn-check" name="prioridad" id="prioridad-urgente" value="URGENTE"
                       autocomplete="off" <?= (($_POST['prioridad'] ?? 'NORMAL') === 'URGENTE') ? 'checked' : ''; ?>>
                <label class="btn btn-outline-danger btn-sm" for="prioridad-urgente">URGENTE</label>
              </div>
            </div>

            <div class="col-md-12">
              <label class="form-label">FACTURADO A *</label>
              <input type="text" class="form-control form-control-sm" id="facturado" name="facturado" required>
            </div>
          </div>

          <!-- Detalle -->
          <div id="registro" class="mt-3">
            <hr class="my-2">
            <div class="row g-2 align-items-end">
              <div class="col-md-12">
                <label class="form-label">CHASIS *</label>
                <input class="form-control form-control-sm" id="chasis" autocomplete="off"
                       placeholder="Escanee o pegue varios (espacios, comas o Enter)">
              </div>
            </div>

            <div id="tabla" class="mt-3">
              <table id="vehiculo" class="table table-sm table-hover">
                <thead class="table-light">
                  <tr>
                    <th>Chasis</th>
                    <th>Marca</th>
                    <th>Modelo</th>
                    <th>Color</th>
                    <th>Bodega</th><!-- SOLO aquí en la tabla del modal -->
                    <th class="text-end">Acción</th>
                  </tr>
                </thead>
                <tbody></tbody>
              </table>
            </div>
          </div>
        </div>
        <div class="modal-footer py-2">
          <button onclick="finalizar()" id="btn-finalizar" type="button" class="btn btn-success btn-sm" disabled>
            <i class="fa fa-check me-1"></i> Finalizar requerimiento
          </button>
          <button type="button" class="btn btn-outline-secondary btn-sm" data-bs-dismiss="modal">Cerrar</button>
        </div>
      </div>
    </div>
  </div>

<?php include 'partials/footer.php'; ?>

<link rel="stylesheet" href="https://cdn.datatables.net/2.0.8/css/dataTables.dataTables.min.css">
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.datatables.net/2.0.8/js/dataTables.min.js"></script>

<style>
/* Overlay de guardado dentro del modal */
.modal-saving-overlay {
  position: absolute; inset: 0; background: rgba(255,255,255,.6);
  display: none; align-items: center; justify-content: center; z-index: 10;
}
.modal-saving-overlay.show { display: flex; }
</style>

<script>
// ============ CONFIG GLOBAL ============
document.documentElement.style.fontSize = '15px';
const API_BASE = '/latkic_2.0/views/pages/api/'; // AJUSTA ESTA RUTA A TU ESTRUCTURA
const api = p => API_BASE + p;

// ============ ESTADO GLOBAL ============
window.vehiculosTmp = window.vehiculosTmp || [];
let tablaPDI;
let _savingPDI = false; // flag de guardado en progreso

// ============ HELPERS ============
const esc = s => String(s ?? '').replace(/[&<>"']/g, m => ({
  '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'
}[m]));
function toast(msg, tipo='info'){
  const id='t'+Date.now(), cls={info:'primary',success:'success',warning:'warning',danger:'danger'}[tipo]||'secondary';
  const el=document.createElement('div');
  el.className=`alert alert-${cls} py-2 px-3 small position-fixed end-0 top-0 mt-2 me-2 shadow`;
  el.id=id; el.textContent=msg; document.body.appendChild(el);
  setTimeout(()=>el.remove(),3500);
}
function tokensDeChasis(str) {
  return String(str || '').trim().split(/[\s,;]+/).filter(Boolean).map(s => s.toUpperCase());
}
// Helper genérico para estado de carga en botones
function setBtnLoading(btn, isLoading, loadingText = 'Guardando...') {
  if (!btn) return;
  if (isLoading) {
    btn.dataset.originalHtml = btn.innerHTML;
    btn.disabled = true;
    btn.innerHTML = `
      <span class="spinner-border spinner-border-sm me-1" role="status" aria-hidden="true"></span>
      ${loadingText}
    `;
  } else {
    btn.disabled = (window.vehiculosTmp?.length ?? 0) === 0;
    if (btn.dataset.originalHtml) btn.innerHTML = btn.dataset.originalHtml;
  }
}

// ============ CACHE LOCAL ============
const CACHE_KEY = 'PDI_TMP_CACHE_v1';
const cacheLoad = ()=>{ try{ const r=localStorage.getItem(CACHE_KEY); return r?JSON.parse(r):null; }catch{ return null; } };
const cacheSave = p=>{ try{ localStorage.setItem(CACHE_KEY, JSON.stringify(p)); }catch(e){ console.warn(e); } };
const cacheClear = ()=> localStorage.removeItem(CACHE_KEY);

function syncCacheFromUI() {
  const header = {
    alistamiento: (document.getElementById('alistamiento')?.value || '').trim(),
    requerido:    (document.getElementById('requerido')?.value || '').trim(),
    lugar:        (document.getElementById('lugar')?.value || '').trim(),
    f_req:        (document.getElementById('f_requerido')?.value || '').trim(),
    f_ent:        (document.getElementById('f_entrega')?.value || '').trim(),
    prioridad:    (document.querySelector('input[name="prioridad"]:checked')?.value || 'NORMAL'),
    facturado:    (document.getElementById('facturado')?.value || '').trim()
  };
  const existing = cacheLoad() || { header:{}, vehiculos:[] };
  existing.header = header;
  existing.vehiculos = [...window.vehiculosTmp];
  cacheSave(existing);
}
function syncUIFromCache() {
  const c = cacheLoad();
  if (!c) { window.vehiculosTmp = []; renderTablaVehiculos(); return; }
  if (c.header) {
    const h=c.header;
    if (document.getElementById('alistamiento')) document.getElementById('alistamiento').value = h.alistamiento || '';
    if (document.getElementById('requerido'))    document.getElementById('requerido').value    = h.requerido || '';
    if (document.getElementById('lugar'))        document.getElementById('lugar').value        = h.lugar || '';
    if (document.getElementById('f_requerido'))  document.getElementById('f_requerido').value  = h.f_req || '';
    if (document.getElementById('f_entrega'))    document.getElementById('f_entrega').value    = h.f_ent || '';
    if (document.getElementById('facturado'))    document.getElementById('facturado').value    = h.facturado || '';
    if (h.prioridad) {
      const r = document.getElementById(h.prioridad === 'URGENTE' ? 'prioridad-urgente' : 'prioridad-normal');
      if (r) r.checked = true;
    }
  }
  window.vehiculosTmp = Array.isArray(c.vehiculos) ? c.vehiculos : [];
  renderTablaVehiculos();
}

// ============ Opciones de tb_tipolavada (para columna select) ============
let LAVADAS_OPTS = [];  // [{id, text}]
async function cargarLavadas() {
  if (LAVADAS_OPTS.length) return LAVADAS_OPTS;
  const r = await fetch(api('tipolavada_list.php'));
  const j = await r.json().catch(()=>null);
  if (!j?.success) throw new Error(j?.message || 'No se pudo cargar tipos de lavada');
  // Espera {success:true, options:[{id,text}]}. Si tu endpoint devuelve otra forma, mapea aquí.
  LAVADAS_OPTS = j.options || [];
  return LAVADAS_OPTS;
}
function renderSelectLavada(idpdi, idtipo_lavada) {
  const opts = LAVADAS_OPTS;
  const current = idtipo_lavada ? Number(idtipo_lavada) : null;
  const htmlOpts = ['<option value="">-- Seleccione --</option>']
    .concat(opts.map(o => `<option value="${o.id}" ${current===o.id?'selected':''}>${esc(o.text)}</option>`));
  return `<select class="form-select form-select-sm lavada-sel" data-idpdi="${idpdi}" style="max-width:220px">${htmlOpts.join('')}</select>`;
}
async function guardarLavada(idpdi, idtipo) {
  const r = await fetch(api('lavada_save_strict.php'), {
    method: 'POST',
    headers: {'Content-Type':'application/json'},
    body: JSON.stringify({ idpdi, idtipo_lavada: (idtipo?Number(idtipo):null) })
  });
  const j = await r.json().catch(()=>null);
  if (!j?.success) throw new Error(j?.message || 'No se pudo guardar la lavada');
  tablaPDI?.ajax?.reload(null, false);
  toast(
    j.action === 'insert' ? 'Lavada creada'
    : j.action === 'update' ? 'Lavada actualizada'
    : 'Sin cambios'
  , 'success');
  return j;
}


// ============ INICIO ============
document.addEventListener('DOMContentLoaded', () => {
  initTablaPDI();

  // Modal: restaurar caché
  const modalEl = document.getElementById('agregar_factura');
  modalEl.addEventListener('show.bs.modal', () => {
    syncUIFromCache();
    document.getElementById('btn-finalizar').disabled = window.vehiculosTmp.length === 0;
    document.getElementById('alistamiento').focus();
  });

  // Entrada chasis: Enter/Espacio/pegar
  const chInp = document.getElementById('chasis');
  chInp.addEventListener('keydown', (e) => {
    if (e.key === 'Enter' || e.key === ' ') {
      e.preventDefault();
      agregarVariosDesdeInput_cacheOnly();
    }
  });
  chInp.addEventListener('paste', () => setTimeout(agregarVariosDesdeInput_cacheOnly, 0));

  // Cambios de cabecera -> cache
  ['alistamiento','requerido','lugar','f_requerido','f_entrega','facturado'].forEach(id=>{
    const el=document.getElementById(id); if(!el) return;
    el.addEventListener('change', syncCacheFromUI);
    el.addEventListener('input', ()=>{ clearTimeout(el._t); el._t=setTimeout(syncCacheFromUI,250); });
  });
  document.querySelectorAll('input[name="prioridad"]').forEach(r=>{
    r.addEventListener('change', syncCacheFromUI);
  });
});

// ============ DataTable ============
async function initTablaPDI(){
  await cargarLavadas(); // asegura tener opciones antes de renderizar

  tablaPDI = new DataTable('#pdi', {
    processing: true,                                  // spinner mientras carga/recarga
    ajax: { url: api('alistamiento_list.php'), dataSrc: 'data' },
    columns: [
      { data: 'marca' },
      { data: 'modelo' },
      { data: 'color' },
      { data: 'chasis' },
      { data: 'lugar' },
      { data: 'alistamiento' },
      { data: 'fecha_req' },
      { data: 'fecha_ent' },
      { 
        data: 'prioridad',
        render: d => d==='URGENTE'
          ? `<span class="badge text-bg-danger">${esc(d)}</span>`
          : `<span class="badge text-bg-secondary">${esc(d||'NORMAL')}</span>`
      },
      { // NUEVA: select Tipo de lavada
        data: null,
        orderable: false,
        className: 'align-middle',
        render: (row) => renderSelectLavada(row.idpdi, row.idtipo_lavada ?? null)
      },
      { data: 'facturado' }
    ],
    pageLength: 10,
    order: [[6, 'desc']], // por fecha_req (índice 6)
    language: { url: 'https://cdn.datatables.net/plug-ins/2.0.8/i18n/es-ES.json' }
  });

  // Delegación para cambios en el select
  document.querySelector('#pdi tbody').addEventListener('change', async (ev) => {
    const sel = ev.target.closest('.lavada-sel');
    if (!sel) return;
    const idpdi = Number(sel.dataset.idpdi || 0);
    const val = sel.value;           // '' o id
    const prev = sel.getAttribute('data-old') ?? ''; // valor previo

    sel.disabled = true;
    try {
      await guardarLavada(idpdi, val);
      sel.setAttribute('data-old', val);
      // Sin toast aquí: guardarLavada ya notifica
    } catch (e) {
      sel.value = prev; // revertir si falla
      toast(e.message || 'Error al actualizar', 'danger');
    } finally {
      sel.disabled = false;
    }
  });
}

// ============ Buscar chasis ============
async function buscarDatadaiLote(chArr) {
  const res = await fetch(api('datadai_por_chasis.php'), {
    method: 'POST',
    headers: {'Content-Type':'application/json'},
    body: JSON.stringify({ chasis_multi: chArr, solo_simple: 1, estado_max: 20 })
  });
  const js = await res.json().catch(()=>null);
  if (!js?.success) throw new Error(js?.message || 'Error buscando chasis');
  return js; // {found:[{iddatadai,chasis,marca,modelo,color,bodega}], not_found:[...]}
}

async function agregarVariosDesdeInput_cacheOnly() {
  const input = document.getElementById('chasis');
  const toks = tokensDeChasis(input.value);
  if (toks.length === 0) { toast('Ingrese o pegue uno o más chasis.', 'warning'); return; }

  const ya = new Set(window.vehiculosTmp.map(v => v.chasis.toUpperCase()));
  const pedir = [...new Set(toks)].filter(ch => !ya.has(ch));
  if (pedir.length === 0) { toast('Todos los chasis ya están en la tabla.', 'warning'); return; }

  try {
    const js = await buscarDatadaiLote(pedir);
    const found = Array.isArray(js.found) ? js.found : [];
    for (const v of found) {
      window.vehiculosTmp.push({
        iddatadai: v.iddatadai,
        chasis: (v.chasis || '').toUpperCase(),
        marca:  v.marca  || '',
        modelo: v.modelo || '',
        color:  v.color  || '',
        bodega: v.bodega || ''   // <- MOSTRAR SOLO EN TABLA DEL MODAL
      });
    }
    renderTablaVehiculos();
    input.value = '';
    syncCacheFromUI();

    if (found.length) toast(`Agregados ${found.length} a la tabla (en caché).`, 'success');
    if (Array.isArray(js.not_found) && js.not_found.length) {
      toast(`No encontrados: ${js.not_found.slice(0,3).join(', ')}${js.not_found.length>3?'...':''}`, 'warning');
    }
  } catch (e) {
    toast(e.message, 'danger');
  }
}

function quitarVehiculo(ch) {
  window.vehiculosTmp = window.vehiculosTmp.filter(v => v.chasis !== ch);
  renderTablaVehiculos();
  syncCacheFromUI();
}
window.quitarVehiculo = quitarVehiculo;

function renderTablaVehiculos() {
  const tb = document.querySelector('#vehiculo tbody');
  if (!tb) return;
  tb.innerHTML = '';
  for (const v of window.vehiculosTmp) {
    const tr = document.createElement('tr');
    tr.innerHTML = `
      <td>${esc(v.chasis)}</td>
      <td>${esc(v.marca)}</td>
      <td>${esc(v.modelo)}</td>
      <td>${esc(v.color)}</td>
      <td>${esc(v.bodega)}</td>
      <td class="text-end">
        <button class="btn btn-outline-danger btn-sm btn-quitar" title="Quitar" data-ch="${esc(v.chasis)}">
          <i class="fa fa-trash"></i>
        </button>
      </td>`;
    tb.appendChild(tr);
  }
  tb.querySelectorAll('.btn-quitar').forEach(btn=>{
    btn.onclick = e => { 
      const ch = e.currentTarget.getAttribute('data-ch') || '';
      quitarVehiculo(ch);
    };
  });
  const btnFin = document.getElementById('btn-finalizar');
  if (btnFin) btnFin.disabled = (window.vehiculosTmp.length === 0);
}

// ============ Guardado en tb_pdi ============
function toMySQLDateTime(s) {
  if (!s) return '';
  s = String(s).trim().replace('T',' ');
  return /^\d{4}-\d{2}-\d{2} \d{2}:\d{2}$/.test(s) ? (s+':00') : s;
}
function validarCabeceraParaFinalizar() {
  const alistamiento = (document.getElementById('alistamiento')?.value || '').trim();
  const requerido    = (document.getElementById('requerido')?.value || '').trim();
  const lugar        = (document.getElementById('lugar')?.value || '').trim();
  const f_req_raw    = (document.getElementById('f_requerido')?.value || '').trim();
  const f_ent_raw    = (document.getElementById('f_entrega')?.value || '').trim();
  const prioridad    = (document.querySelector('input[name="prioridad"]:checked')?.value || 'NORMAL');
  const facturado    = (document.getElementById('facturado')?.value || '').trim();

  if (!alistamiento) throw new Error('Seleccione el tipo de alistamiento.');
  if (!requerido)    throw new Error('Ingrese “Requerido por”.');
  if (!lugar)        throw new Error('Seleccione el Lugar.');
  if (!f_req_raw)    throw new Error('Seleccione la fecha de requerimiento.');
  if (!f_ent_raw)    throw new Error('Seleccione la fecha aproximada de entrega.');
  if (f_ent_raw < f_req_raw) throw new Error('La fecha de entrega no puede ser menor a la de requerimiento.');
  if (!facturado)    throw new Error('Ingrese “Facturado a”.');

  return {
    alistamiento,
    requerido,
    lugar,
    requerimiento: toMySQLDateTime(f_req_raw),
    entrega:       toMySQLDateTime(f_ent_raw),
    prioridad,
    facturado
  };
}

async function finalizar() {
  if (_savingPDI) return; // evita doble click
  const btn = document.getElementById('btn-finalizar');
  const overlay = document.getElementById('savingOverlay');

  try {
    if (!Array.isArray(window.vehiculosTmp) || window.vehiculosTmp.length === 0) {
      throw new Error('Agregue al menos un vehículo.');
    }
    const header = validarCabeceraParaFinalizar();

    // Bloqueo UI
    _savingPDI = true;
    setBtnLoading(btn, true, 'Guardando...');
    overlay?.classList.add('show');

    const r = await fetch(api('requerimiento_guardar.php'), {
      method:'POST',
      headers:{'Content-Type':'application/json'},
      body: JSON.stringify({ header, vehiculos: window.vehiculosTmp })
    });
    const txt = await r.text();
    let j = null; try { j = JSON.parse(txt); } catch {}

    if (!r.ok || !j || j.success !== true) {
      const msg = j?.message || j?.sql_error || 'No se pudo crear el requerimiento.';
      throw new Error(msg);
    }

    // Limpiar y recargar
    cacheClear();
    window.vehiculosTmp = [];
    renderTablaVehiculos();
    toast(`Requerimiento guardado (${j.insertados} filas${j.saltados?`, ${j.saltados} saltados`:''}).`, 'success');
    tablaPDI?.ajax?.reload(null, false);
    bootstrap.Modal.getInstance(document.getElementById('agregar_factura'))?.hide();
  } catch (e) {
    console.error('[finalizar] error:', e);
    toast(e.message || 'Error guardando requerimiento', 'danger');
  } finally {
    _savingPDI = false;
    setBtnLoading(btn, false);
    overlay?.classList.remove('show');
  }
}
window.finalizar = finalizar;
</script>

<script>
// Prefill seguro al abrir el modal (usuario + límites de fecha)
document.addEventListener('shown.bs.modal', function (ev) {
  if (ev.target && ev.target.id === 'agregar_factura') {
    const reqInp = document.getElementById('requerido');
    if (reqInp && !reqInp.value) { reqInp.value = '<?= addslashes($usuario) ?>'; }

    const fReq = document.getElementById('f_requerido');
    const fEnt = document.getElementById('f_entrega');
    if (fReq && !fReq.value) fReq.value = '<?= $hoyFechaHora ?>';
    if (fReq) fReq.min = '<?= $hoyFechaHora ?>';
    if (fEnt && !fEnt.value) fEnt.value = '<?= $manianaFechaH ?>';
    if (fEnt) fEnt.min = '<?= $hoyFechaHora ?>';
  }
});
</script>

<script>
  // Cargar opciones de "Lugar" al iniciar (y permitir preselección si aplica)
  async function cargarLugares(preseleccion = '') {
    try {
      const sel = document.getElementById('lugar');
      if (!sel) return;

      // Limpia dejando placeholder
      sel.innerHTML = '<option value="">Seleccione…</option>';

      const r = await fetch(new URL('api/lugares_list.php', window.location.href).href);
      const j = await r.json();
      if (!j?.success) throw new Error(j?.error || 'No se pudo cargar lugares');

      const frag = document.createDocumentFragment();
      for (const it of (j.data || [])) {
        const opt = document.createElement('option');
        opt.value = String(it.id);
        opt.textContent = String(it.nombre);
        frag.appendChild(opt);
      }
      sel.appendChild(frag);

      // Si estás editando un registro y quieres marcar el valor guardado:
      if (preseleccion) {
        sel.value = String(preseleccion);
      }
    } catch (e) {
      console.error(e);
      if (typeof Swal !== 'undefined') {
        Swal.fire({ icon:'error', title:'No se pudo cargar los lugares', toast:true, position:'top-end', timer:2000, showConfirmButton:false });
      }
    }
  }

  // Llama al cargar la página (si tienes un valor a preseleccionar, pásalo aquí)
  document.addEventListener('DOMContentLoaded', function(){
    cargarLugares();
  });
</script>

</body>
</html>
