<?php
// movimientos.php — SGO · Inventario · Movimientos
declare(strict_types=1);
session_start();
if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES') {
  header("Location: ../../views/index.php"); exit();
}
$usuario = $_SESSION['usuario'] ?? '';
$permiso = (int)($_SESSION['permiso'] ?? 0);
?>
<?php include __DIR__ . '/partials/header.php'; ?>
<!doctype html>
<html lang="es" data-theme="light">
<head>
  <meta charset="utf-8"/>
  <meta name="viewport" content="width=device-width, initial-scale=1"/>
  <title>SGO · Inventario · Movimientos</title>

  <!-- CSS (quita si ya están en header.php) -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.datatables.net/1.13.8/css/dataTables.bootstrap5.min.css" rel="stylesheet">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" referrerpolicy="no-referrer" />
  <link href="https://cdn.jsdelivr.net/npm/@sweetalert2/theme-bootstrap-4@5/bootstrap-4.min.css" rel="stylesheet">

  <style>
    :root { font-size: 15.5px; }
    .dt-center { text-align:center; }
    html, body, .container-fluid, .row { height: 100%; }
    .flex-fill { min-height: 0; }
    #workArea, #tableWrap { min-height: 0; padding: 0; }
    .dataTables_wrapper { padding: 0; }
    html[data-theme="dark"] body { background:#0f1115; color:#e7e7e7; }
    html[data-theme="dark"] .table { color:#e7e7e7; }
    html[data-theme="dark"] .table thead th { color:#f0f0f0; }
    .mono { font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", monospace; }
    .renglon .form-select, .renglon .form-control { padding-top:.25rem; padding-bottom:.25rem; }
  </style>
</head>
<body>
<div class="container-fluid">
  <div class="row">
    <?php include __DIR__ . '/partials/menu.php'; ?>

    <main class="col-md-9 ms-sm-auto col-lg-10 px-0 d-flex flex-column">
      <?php include __DIR__ . '/partials/topbar.php'; ?>

      <section id="workArea" class="flex-fill d-flex flex-column">
        <div id="toolbar" class="d-flex align-items-center gap-2 px-3 py-2 border-bottom">
          <h6 class="mb-0"><i class="fa-solid fa-right-left me-2"></i> Movimientos de Inventario</h6>
          <input id="txtBuscar" class="form-control form-control-sm" placeholder="Buscar (tipo, bodegas, motivo, ref)">
          <div class="ms-auto d-flex gap-2">
            <button id="btnNuevo" class="btn btn-primary btn-sm"><i class="fa-solid fa-plus"></i> Nuevo</button>
          </div>
        </div>

        <div id="tableWrap" class="flex-fill d-flex flex-column">
          <div class="flex-fill overflow-hidden">
            <table id="tabla" class="table table-striped table-hover w-100 m-0">
              <thead>
                <tr>
                  <th style="width:130px">Fecha</th>
                  <th style="width:70px" class="dt-center">Tipo</th>
                  <th>Origen</th>
                  <th>Destino</th>
                  <th>Motivo</th>
                  <th class="dt-center" style="width:70px">Items</th>
                  <th class="dt-center" style="width:180px">Acciones</th>
                </tr>
              </thead>
              <tbody></tbody>
            </table>
          </div>
        </div>

      </section>
    </main>
  </div>
</div>

<!-- Modal CRUD -->
<div class="modal fade" id="modalForm" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-xl">
    <form class="modal-content" id="formMov">
      <div class="modal-header py-2">
        <h6 class="modal-title"><span id="ttlForm">Nuevo Movimiento</span></h6>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
      </div>
      <div class="modal-body">
        <input type="hidden" id="id_movimiento" value="0">
        <div class="row g-2 mb-2">
          <div class="col-md-3">
            <label class="form-label">Fecha</label>
            <input type="datetime-local" id="fecha" class="form-control">
          </div>
          <div class="col-md-2">
            <label class="form-label">Tipo *</label>
            <select id="tipo" class="form-select" required>
              <option value="">— Seleccione —</option>
              <option value="IN">Entrada</option>
              <option value="OUT">Salida</option>
              <option value="TRF">Transferencia</option>
            </select>
          </div>
          <div class="col-md-3">
            <label class="form-label">Bodega Origen</label>
            <select id="id_bodega_origen" class="form-select"></select>
          </div>
          <div class="col-md-3">
            <label class="form-label">Bodega Destino</label>
            <select id="id_bodega_destino" class="form-select"></select>
          </div>
          <div class="col-md-6">
            <label class="form-label">Motivo</label>
            <input id="motivo" class="form-control">
          </div>
          <div class="col-md-3">
            <label class="form-label">Ref. Tipo</label>
            <input id="ref_tipo" class="form-control" placeholder="OC, FAC, AJUSTE...">
          </div>
          <div class="col-md-3">
            <label class="form-label">Ref. ID</label>
            <input id="ref_id" class="form-control" placeholder="# documento">
          </div>
        </div>

        <div class="d-flex align-items-center justify-content-between mb-2">
          <h6 class="mb-0">Detalle</h6>
          <button type="button" class="btn btn-sm btn-outline-primary" id="btnAddRow"><i class="fa-solid fa-plus"></i> Renglón</button>
        </div>

        <div id="detalle" class="border rounded p-2">
          <!-- renglones se insertan aquí -->
        </div>
      </div>
      <div class="modal-footer py-2">
        <button type="button" class="btn btn-secondary btn-sm" data-bs-dismiss="modal">Cancelar</button>
        <button type="submit" class="btn btn-primary btn-sm">Guardar</button>
      </div>
    </form>
  </div>
</div>

<?php include __DIR__ . '/partials/footer.php'; ?>

<!-- JS -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.datatables.net/1.13.8/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.8/js/dataTables.bootstrap5.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<script>
// ===== Helper fetchJSON: evita 'Unexpected token <' mostrando el HTML del error si la API no devuelve JSON =====
async function fetchJSON(url, options) {
  const res = await fetch(url, options);
  const txt = await res.text();
  if (!res.ok) {
    console.error('HTTP error', res.status, txt);
    Swal.fire('Error HTTP ' + res.status, txt.slice(0, 400), 'error');
    throw new Error('HTTP ' + res.status);
  }
  try {
    return JSON.parse(txt);
  } catch (e) {
    console.error('Respuesta NO-JSON desde', url, txt);
    Swal.fire('Error del servidor', txt.slice(0, 400), 'error');
    throw new Error('Servidor devolvió NO-JSON');
  }
}
</script>

<script>
const apiBase = (window.SGO_API_BASE ?? 'api/');
let dt=null, modal, form;
let REF = { accesorios:[], ubicaciones:[] }; // bodegas se gestionan vía endpoint dedicado

document.addEventListener('DOMContentLoaded', () => {
  modal = new bootstrap.Modal(document.getElementById('modalForm'));
  form  = document.getElementById('formMov');

  initTable();
  window.addEventListener('resize', debounce(resizeTableHeight, 150));

  document.getElementById('txtBuscar').addEventListener('input', recargarTabla);
  document.getElementById('btnNuevo').addEventListener('click', nuevo);
  document.getElementById('btnAddRow').addEventListener('click', addRow);
  document.getElementById('tipo').addEventListener('change', onTipoChange);
  document.getElementById('id_bodega_origen').addEventListener('change', onOrigenChange);
  form.addEventListener('submit', onGuardar);

  cargarRefData(); // accesorios + ubicaciones base
});

/* ===== DataTable ===== */
function initTable(){
  if (dt) return;
  const h = computeScrollY();
  dt = new DataTable('#tabla', {
    data: [],
    columns: [
      { data: 'fecha', render: v => v ? v.replace('T',' ') : '' },
      { data: 'tipo', className:'dt-center', render: t => t==='IN'?'Entrada':(t==='OUT'?'Salida':'Transfer') },
      { data: 'bodega_origen' },
      { data: 'bodega_destino' },
      { data: 'motivo' },
      { data: 'items', className:'dt-center' },
      { data: null, className:'dt-center', orderable:false, render: row => accionesHTML(row) }
    ],
    order: [[0,'desc']],
    scrollY: h + 'px',
    scrollCollapse: true,
    paging: true,
    language: { url: 'https://cdn.datatables.net/plug-ins/1.13.8/i18n/es-ES.json' }
  });
  recargarTabla();
}
function accionesHTML(row){
  const id = row.id_movimiento;
  return `
    <button class="btn btn-sm btn-outline-primary me-1" onclick="editar(${id})" title="Editar"><i class="fa-solid fa-pen"></i></button>
    <button class="btn btn-sm btn-outline-danger" onclick="eliminar(${id})" title="Eliminar"><i class="fa-solid fa-trash"></i></button>
  `;
}
function recargarTabla(){
  const q = encodeURIComponent(document.getElementById('txtBuscar').value || '');
  fetchJSON(`${apiBase}movimiento_list.php?_=${Date.now()}&search=${q}`)
    .then(j=>{
      if(!j.success) throw new Error(j.message||'Error');
      const page = dt.page();
      dt.clear(); dt.rows.add(j.data||[]); dt.draw(false);
      dt.page(page).draw(false);
      resizeTableHeight();
    }).catch(err=>{
      console.error(err);
      Swal.fire('Error','No se pudo cargar la lista','error');
    });
}
function resizeTableHeight(){
  if (!dt) return;
  const h = computeScrollY();
  const wrapper = document.querySelector('#tabla')?.closest('.dataTables_wrapper');
  const scroller = wrapper?.querySelector('.dataTables_scrollBody');
  if (scroller) { scroller.style.height = h + 'px'; scroller.style.maxHeight = h + 'px'; }
  dt.columns.adjust();
}
function computeScrollY(){
  const tableWrap = document.getElementById('tableWrap');
  const wrapper   = document.querySelector('#tabla')?.closest('.dataTables_wrapper');
  const headH = wrapper?.querySelector('.dataTables_scrollHead')?.offsetHeight || 0;
  const infoH = wrapper?.querySelector('.dataTables_info')?.offsetHeight || 0;
  const pagH  = wrapper?.querySelector('.dataTables_paginate')?.offsetHeight || 0;
  const gap   = 16;
  const wrapH = tableWrap?.clientHeight || 400;
  return Math.max(160, wrapH - headH - infoH - pagH - gap);
}

/* ===== RefData / Combos ===== */
// Accesorios/ubicaciones base
function cargarRefData(){
  fetchJSON(`${apiBase}mov_refdata.php?_=${Date.now()}`)
    .then(j=>{
      if(!j.success) throw new Error('refdata');
      REF.accesorios  = j.accesorios || [];
      REF.ubicaciones = j.ubicaciones || [];
    }).catch(e=>console.error(e));
}

// Cargar bodegas según tipo (reglas de negocio)
async function cargarBodegasPorTipo(){
  const tipo = document.getElementById('tipo').value;
  const selO = document.getElementById('id_bodega_origen');
  const selD = document.getElementById('id_bodega_destino');
  const prevO = selO.value, prevD = selD.value;

  // Origen
  let urlOrigen = `${apiBase}bodega_origen.php?`;
  const onlyWithStockOrigen = (tipo==='OUT' || tipo==='TRF') ? 1 : 0;
  urlOrigen += `onlyWithStock=${onlyWithStockOrigen}`;
  const listOrigen = await fetchJSON(urlOrigen).then(j=>j.data||[]).catch(()=>[]);

  selO.innerHTML = '<option value="">— Origen —</option>';
  listOrigen.forEach(b=> selO.append(new Option(`${b.nombre}${b.ciudad? ' ('+b.ciudad+')':''}`, b.id_bodega)));

  // Destino
  let urlDestino = `${apiBase}bodega_origen.php?onlyWithStock=0`;
  const listDestino = await fetchJSON(urlDestino).then(j=>j.data||[]).catch(()=>[]);
  selD.innerHTML = '<option value="">— Destino —</option>';
  listDestino.forEach(b=> selD.append(new Option(`${b.nombre}${b.ciudad? ' ('+b.ciudad+')':''}`, b.id_bodega)));

  if ([...selO.options].some(o=>o.value===prevO)) selO.value = prevO;
  if ([...selD.options].some(o=>o.value===prevD)) selD.value = prevD;
}

/* ===== UI Form ===== */
function nuevo(){
  form.reset();
  document.getElementById('id_movimiento').value = 0;
  document.getElementById('ttlForm').textContent = 'Nuevo Movimiento';
  document.getElementById('detalle').innerHTML = '';
  addRow();
  cargarBodegasPorTipo().then(()=> onTipoChange());
  modal.show();
}
function editar(id){
  fetchJSON(`${apiBase}movimiento_get.php?id=${id}`)
    .then(async j=>{
      if(!j.success) return Swal.fire('Atención', j.message||'No encontrado', 'warning');
      const c = j.data.cab, d = j.data.det || [];
      document.getElementById('id_movimiento').value = c.id_movimiento;
      document.getElementById('fecha').value = (c.fecha||'').replace(' ','T');
      document.getElementById('tipo').value = c.tipo;

      await cargarBodegasPorTipo();
      document.getElementById('id_bodega_origen').value = c.id_bodega_origen || '';
      document.getElementById('id_bodega_destino').value = c.id_bodega_destino || '';

      document.getElementById('motivo').value = c.motivo || '';
      document.getElementById('ref_tipo').value = c.ref_tipo || '';
      document.getElementById('ref_id').value = c.ref_id || '';
      onTipoChange();

      const cont = document.getElementById('detalle');
      cont.innerHTML = '';
      d.forEach(row => addRow(row));
      if (d.length===0) addRow();

      if (c.id_bodega_origen) await cargarUbicacionesPorBodega(c.id_bodega_origen);

      document.getElementById('ttlForm').textContent = 'Editar Movimiento';
      modal.show();
    })
    .catch(()=> Swal.fire('Error','No se pudo obtener el registro','error'));
}
function onTipoChange(){
  const tipo = document.getElementById('tipo').value;
  const o = document.getElementById('id_bodega_origen');
  const d = document.getElementById('id_bodega_destino');
  o.disabled = (tipo==='IN');
  d.disabled = (tipo==='OUT');
  cargarBodegasPorTipo();
}
function onOrigenChange(e){
  const bod = parseInt(e.target.value || '0');
  if (!bod){
    document.querySelectorAll('#detalle .sel-ubi').forEach(s=> s.innerHTML = '<option value="">— Ubicación —</option>');
    return;
  }
  cargarUbicacionesPorBodega(bod);
}

async function cargarUbicacionesPorBodega(id_bod){
  const data = await fetchJSON(`${apiBase}ubicaciones_por_bodega.php?id_bodega=${id_bod}`)
                      .then(j=>j.data||[])
                      .catch(()=>[]);
  document.querySelectorAll('#detalle .sel-ubi').forEach(s=>{
    const cur = s.value;
    s.innerHTML = '<option value="">— Ubicación —</option>';
    data.forEach(u => s.append(new Option(`${u.codigo} ${u.nombre}`, u.id_ubicacion)));
    if (cur) s.value = cur;
  });
}

/* Renglones */
function addRow(pref=null){
  const div = document.createElement('div');
  div.className = 'renglon row g-2 align-items-center mb-1';
  div.innerHTML = `
    <div class="col-md-6">
      <select class="form-select sel-acc" required>
        <option value="">— Accesorio —</option>
      </select>
    </div>
    <div class="col-md-2">
      <input type="number" min="0.01" step="0.01" class="form-control inp-cant" placeholder="Cant." required>
    </div>
    <div class="col-md-3">
      <select class="form-select sel-ubi">
        <option value="">— Ubicación —</option>
      </select>
    </div>
    <div class="col-md-1 text-end">
      <button type="button" class="btn btn-sm btn-outline-danger" onclick="this.closest('.renglon').remove()"><i class="fa-solid fa-xmark"></i></button>
    </div>
  `;
  document.getElementById('detalle').appendChild(div);

  const sAcc = div.querySelector('.sel-acc');
  (REF.accesorios||[]).forEach(a => sAcc.append(new Option(a.nombre, a.id_accesorio)));

  const bod = parseInt(document.getElementById('id_bodega_origen').value || '0');
  if (bod){
    cargarUbicacionesPorBodega(bod);
  } else {
    const sUbi = div.querySelector('.sel-ubi');
    sUbi.innerHTML = '<option value="">— Ubicación —</option>';
  }

  if (pref){
    sAcc.value = pref.id_accesorio;
    div.querySelector('.inp-cant').value = pref.cantidad;
    if (pref.id_ubicacion) div.querySelector('.sel-ubi').value = pref.id_ubicacion;
  }
}

function onGuardar(e){
  e.preventDefault();
  const id  = parseInt(document.getElementById('id_movimiento').value || '0');
  const tipo= document.getElementById('tipo').value;
  const id_o= parseInt(document.getElementById('id_bodega_origen').value || '0');
  const id_d= parseInt(document.getElementById('id_bodega_destino').value || '0');

  if (!['IN','OUT','TRF'].includes(tipo)) return Swal.fire('Validación','Selecciona un tipo válido','info');
  if (tipo==='IN'  && !id_d) return Swal.fire('Validación','Destino requerido para Entrada','info');
  if (tipo==='OUT' && !id_o) return Swal.fire('Validación','Origen requerido para Salida','info');
  if (tipo==='TRF' && (!id_o || !id_d || id_o===id_d)) return Swal.fire('Validación','Origen y Destino válidos para Transferencia','info');

  const filas = [...document.querySelectorAll('#detalle .renglon')];
  if (filas.length===0) return Swal.fire('Validación','Agrega al menos un renglón','info');

  const det = [];
  for (const f of filas){
    const id_accesorio = parseInt(f.querySelector('.sel-acc').value || '0');
    const cantidad     = parseFloat(f.querySelector('.inp-cant').value || '0');
    const id_ubicacion = parseInt(f.querySelector('.sel-ubi').value || '0') || null;
    if (!id_accesorio || !(cantidad>0)) return Swal.fire('Validación','Revisa los renglones (accesorio/cantidad)','info');
    det.push({ id_accesorio, cantidad, id_ubicacion });
  }

  const payload = {
    id_movimiento : id,
    fecha         : document.getElementById('fecha').value || '',
    tipo,
    id_bodega_origen: id_o || null,
    id_bodega_destino: id_d || null,
    motivo: document.getElementById('motivo').value || '',
    ref_tipo: document.getElementById('ref_tipo').value || '',
    ref_id: document.getElementById('ref_id').value || '',
    det
  };

  fetchJSON(`${apiBase}movimiento_save.php`, {
    method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify(payload)
  })
  .then(j=>{
    if(!j.success) return Swal.fire('Atención', j.message||'No se pudo guardar', 'warning');
    Swal.fire('Éxito', j.message||'Guardado', 'success');
    modal.hide();
    recargarTabla();
  })
  .catch(err=> Swal.fire('Error', (err?.message || 'No se pudo guardar'), 'error'));
}

function eliminar(id){
  Swal.fire({
    title:'¿Eliminar?', text:'Se eliminará cabecera y detalle (y se revertirá stock).',
    icon:'warning', showCancelButton:true, confirmButtonText:'Eliminar', cancelButtonText:'Cancelar'
  }).then(res=>{
    if(!res.isConfirmed) return;
    fetchJSON(`${apiBase}movimiento_delete.php`, {
      method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify({id_movimiento:id})
    })
    .then(j=>{
      if(!j.success) return Swal.fire('Atención', j.message||'No se pudo eliminar', 'warning');
      Swal.fire('Listo', j.message||'Eliminado', 'success');
      recargarTabla();
    })
    .catch(()=> Swal.fire('Error','No se pudo eliminar','error'));
  });
}

/* Utils */
function debounce(fn, wait=200){ let t; return (...a)=>{ clearTimeout(t); t=setTimeout(()=>fn.apply(this,a), wait); }; }
</script>

</body>
</html>
