<?php
// importar/import.php — Procesa Solicitud Previa + Excel (.xlsx)
// Inserta cabecera (tb_solicitudprevia), DAI (tb_daientrada) y detalle (tb_datadai),
// comenzando a insertar desde la FILA 3 del Excel. (Incluye NAVIERA texto libre)
declare(strict_types=1);
session_start();
header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/../conex.php';

/* ===== Autoload Composer (PhpSpreadsheet) ===== */
$autoloadCandidates = [
  __DIR__ . '/../../../vendor/autoload.php',
  __DIR__ . '/../../vendor/autoload.php',
  __DIR__ . '/../vendor/autoload.php',
  __DIR__ . '/vendor/autoload.php'
];
$autoloadFound = false;
foreach ($autoloadCandidates as $p) {
  if (is_file($p)) { require_once $p; $autoloadFound = true; break; }
}
if (!$autoloadFound) {
  echo json_encode(['success'=>false,'message'=>'No se encontró vendor/autoload.php (PhpSpreadsheet).']);
  exit;
}

use PhpOffice\PhpSpreadsheet\IOFactory;

date_default_timezone_set('America/Guayaquil');

/* ===== Helpers ===== */
function parseDecimal($v): string {
  $s = trim((string)$v);
  if ($s==='') return '0';
  if (strpos($s, ',') !== false && strpos($s, '.') !== false) {
    $s = str_replace('.', '', $s);
    $s = str_replace(',', '.', $s);
  } elseif (strpos($s, ',') !== false) {
    $s = str_replace(',', '.', $s);
  }
  return is_numeric($s) ? $s : '0';
}

function tableHasCol(mysqli $conn, string $table, string $col): bool {
  $safeTable = preg_replace('/[^a-zA-Z0-9_]/', '', $table);
  $safeCol   = $conn->real_escape_string($col);
  $sql = "SHOW COLUMNS FROM `$safeTable` LIKE '$safeCol'";
  $res = $conn->query($sql);
  return $res && $res->num_rows > 0;
}

/* ===== Entradas del formulario ===== */
$id_usuario      = (int)($_SESSION['idusuario'] ?? 0);

$id_localidad    = (int)($_POST['id_localidad'] ?? 0); // name="id_localidad"
$codigo          = trim($_POST['codigo'] ?? '');
$anio            = trim($_POST['anio'] ?? '');
$numero          = trim($_POST['numero'] ?? '');
$sp              = trim($_POST['sp'] ?? ''); // CÓDIGO-ANIO-NÚMERO

$codigo_embarque = trim($_POST['codigo_embarque'] ?? '');
$pedido          = trim($_POST['pedido'] ?? '');
$tributo         = trim($_POST['tributo'] ?? '');
$naviera         = trim($_POST['naviera'] ?? '');  // <<<<< NAVIERA (texto libre)
if ($naviera !== '') { $naviera = mb_substr($naviera, 0, 120, 'UTF-8'); }

$ruc             = trim($_POST['ruc'] ?? '');
$razon           = trim($_POST['razon'] ?? '');
$fecha_emision   = trim($_POST['fecha_emision'] ?? '');
$id_categoria    = (int)($_POST['id_categoria'] ?? 0);
$daie            = trim($_POST['daie'] ?? '');
$bodega          = (int)($_POST['bodega'] ?? 0);
$fechae          = trim($_POST['fechae'] ?? '');
$fechaev         = trim($_POST['fechaev'] ?? '');
$idempresa       = (int)($_POST['idempresa'] ?? 0); // hidden
$regimen         = "DAP";

// MONEDA
$id_moneda       = (int)($_POST['idmoneda'] ?? 0); // <<< lee name="idmoneda"

/* ===== Contexto ===== */
$fecha_actual = date('Y-m-d');
$hora_actual  = date('H:i:s');
$ip           = $_SERVER['REMOTE_ADDR'] ?? '';
$actividad    = "REGISTRO DE UNIDADES RÉGIMEN DAP";

/* ===== Archivo ===== */
if (!isset($_FILES['archivo']) || ($_FILES['archivo']['error'] ?? 1) !== 0) {
  echo json_encode(['success'=>false,'message'=>'Archivo no enviado o inválido.']); exit;
}
$nombre_archivo = $_FILES['archivo']['name'] ?? '';
$extension      = strtolower(pathinfo($nombre_archivo, PATHINFO_EXTENSION));
if ($extension !== 'xlsx') {
  echo json_encode(['success'=>false,'message'=>'Sólo se permiten archivos .xlsx']); exit;
}
$archivoTmp = $_FILES['archivo']['tmp_name'] ?? '';

/* ===== Validaciones base ===== */
if (!$id_usuario) { echo json_encode(['success'=>false,'message'=>'Sesión inválida.']); exit; }
if (!$idempresa)  { echo json_encode(['success'=>false,'message'=>'Empresa (idempresa) requerida.']); exit; }
if (!$bodega)     { echo json_encode(['success'=>false,'message'=>'Bodega requerida.']); exit; }
if ($codigo==='' || $anio==='' || $numero==='') {
  echo json_encode(['success'=>false,'message'=>'Código/Año/Número de S.P. incompletos.']); exit;
}
if ($sp==='') { $sp = strtoupper($codigo.'-'.$anio.'-'.$numero); }

// <<< Valida moneda obligatoria
if ($id_moneda <= 0) {
  echo json_encode(['success'=>false,'message'=>'Moneda (idmoneda) requerida.']); exit;
}

/* ===== Parámetros por defecto ===== */
$condicion     = 1;
$temporal      = 1;
$id_estado_def = 1;
$cantidad_def  = 1;
$id_ubic_def   = 1;

/* ===== Transacción ===== */
$conn->set_charset('utf8mb4');
$conn->begin_transaction();

$resumen = ['insertados'=>0, 'duplicados'=>0, 'errores'=>0];

try {
  /* ---- 1) tb_solicitudprevia (con NAVIERA si existe la columna) ---- */
  $hasNaviera = tableHasCol($conn, 'tb_solicitudprevia', 'naviera');

  // Construcción dinámica de INSERT según columnas disponibles
  $cols   = ['anio','numero','codigo','numerosp','fecha','id_empresa','id_categoria','id_estado','cod_embarque','id_localidad','condicion','pedido','tributo'];
  $types  = 'sssssiiisiiss'; // s s s s s  i i i s i i  s s
  $values = [$anio, $numero, $codigo, $sp, $fecha_emision, $idempresa, $id_categoria, $id_estado_def, $codigo_embarque, $id_localidad, $condicion, $pedido, $tributo];

  if ($hasNaviera) {
    $cols[]  = 'naviera';
    $types  .= 's';
    $values[]= $naviera;
  }

  // id_moneda siempre existe en tu flujo actual
  $cols[]   = 'id_moneda';
  $types   .= 'i';
  $values[] = $id_moneda;

  $place = implode(',', array_fill(0, count($cols), '?'));
  $sql1  = "INSERT INTO tb_solicitudprevia (".implode(',', $cols).") VALUES ($place)";
  if (!$stmt1 = $conn->prepare($sql1)) {
    throw new Exception('Prepare tb_solicitudprevia: '.$conn->error);
  }
  $stmt1->bind_param($types, ...$values);
  if (!$stmt1->execute()) {
    throw new Exception('Exec tb_solicitudprevia: '.$stmt1->error);
  }
  $id_solicitudprevia = (int)$conn->insert_id;
  $stmt1->close();

  /* ---- 2) tb_daientrada ---- */
  $sql2 = "INSERT INTO tb_daientrada
            (numerodai, fecha, fecha_v, id_solicitudprevia, temporal, id_bodega, condicion)
           VALUES (?,?,?,?,?,?,?)";
  if (!$stmt2 = $conn->prepare($sql2)) {
    throw new Exception('Prepare tb_daientrada: '.$conn->error);
  }
  $stmt2->bind_param("sssiiii", $daie, $fechae, $fechaev, $id_solicitudprevia, $temporal, $bodega, $condicion);
  if (!$stmt2->execute()) {
    throw new Exception('Exec tb_daientrada: '.$stmt2->error);
  }
  $id_daientrada = (int)$conn->insert_id;
  $stmt2->close();

  /* ---- 3) tb_auditoria ---- */
  $sql3 = "INSERT INTO tb_auditoria
            (fecha, hora, ip, detalle, id_daientrada, id_solicitudprevia, id_usuario)
           VALUES (?,?,?,?,?,?,?)";
  if (!$stmt3 = $conn->prepare($sql3)) {
    throw new Exception('Prepare tb_auditoria: '.$conn->error);
  }
  $stmt3->bind_param("ssssiii", $fecha_actual, $hora_actual, $ip, $actividad, $id_daientrada, $id_solicitudprevia, $id_usuario);
  if (!$stmt3->execute()) {
    throw new Exception('Exec tb_auditoria: '.$stmt3->error);
  }
  $stmt3->close();

  // Referencias en sesión (opcional)
  $_SESSION['id_solicitudprevia'] = $id_solicitudprevia;
  $_SESSION['id_daientrada']      = $id_daientrada;
  $_SESSION['id_bodega']          = $bodega;

  /* ---- 4) Procesar Excel (desde FILA 3) ---- */
  $spreadsheet = IOFactory::load($archivoTmp);
  $sheet       = $spreadsheet->getActiveSheet();
  $data        = $sheet->toArray(null, true, true, true);

  $startRow = 3; // insertar desde la fila 3

  // Duplicados por chasis en DB
  if (!$stmt_check = $conn->prepare("SELECT iddatadai FROM tb_datadai WHERE chasis = ? AND condicion = 1")) {
    throw new Exception('Prepare check chasis: '.$conn->error);
  }

  // Columnas opcionales presentes en tb_datadai
  $has = [];
  foreach (['otras','adiciones','fobunitario','fobitem','transporte','seguro','valoraduana','dai','created_at'] as $c) {
    $has[$c] = tableHasCol($conn, 'tb_datadai', $c);
  }

  // Columnas base
  $colsBase = [
    'id_daientrada', 'item','descripcion','modelo','marca','chasis','motor','color',
    'id_solicitudprevia','id_estado','cantidad','condicion','id_ubicacion','id_bodega'
  ];
  $typesBase = 'isssssss'.'iiiiii'; // i + 7s + 6i

  // Opcionales
  $optMap = [
    'otras'        => 's',
    'adiciones'    => 's',
    'fobunitario'  => 's',
    'fobitem'      => 's',
    'transporte'   => 's',
    'seguro'       => 's',
    'valoraduana'  => 's',
    'dai'          => 's',
  ];

  $cols  = $colsBase;
  $types = $typesBase;
  foreach ($optMap as $c => $t) {
    if ($has[$c]) { $cols[] = $c; $types .= $t; }
  }
  if ($has['created_at']) { $cols[] = 'created_at'; $types .= 's'; }

  $placeholders = implode(',', array_fill(0, count($cols), '?'));
  $sqlIns = "INSERT INTO tb_datadai (".implode(',', $cols).") VALUES ($placeholders)";
  if (!$stmtIns = $conn->prepare($sqlIns)) {
    throw new Exception('Prepare tb_datadai: '.$conn->error.' | SQL: '.$sqlIns);
  }

  $visto = [];

  foreach ($data as $rowIndex => $fila) {
    if ((int)$rowIndex < $startRow) continue;

    // Mapeo columnas Excel (ajústalo a tu layout real si difiere)
    $item        = trim((string)($fila['A']  ?? ''));
    $descripcion = trim((string)($fila['F']  ?? ''));
    $modelo      = trim((string)($fila['N']  ?? ''));
    $marca       = trim((string)($fila['O']  ?? ''));
    $otrasX      = trim((string)($fila['Q']  ?? ''));
    $adicionesX  = trim((string)($fila['AC'] ?? ''));
    $fobunit     = parseDecimal($fila['AD'] ?? '0');
    $fobitem     = parseDecimal($fila['AE'] ?? '0');
    $transporte  = parseDecimal($fila['AF'] ?? '0');
    $seguro      = parseDecimal($fila['AG'] ?? '0');
    $valoraduana = parseDecimal($fila['AI'] ?? '0');
    $chasis      = trim((string)($fila['AS'] ?? ''));
    $motor       = trim((string)($fila['AT'] ?? ''));
    $color       = trim((string)($fila['AV'] ?? ''));
    // Extraer los últimos 4 caracteres
    $ultimos4 = str_pad(substr($otrasX, -4), 4, "0", STR_PAD_LEFT);

    // Reglas mínimas
    if ($chasis === '' || $marca === '') { $resumen['errores']++; continue; }

    // Duplicado en archivo
    if (isset($visto[$chasis])) { $resumen['duplicados']++; continue; }

    // Duplicado en BD
    $stmt_check->bind_param("s", $chasis);
    $stmt_check->execute();
    $stmt_check->store_result();
    if ($stmt_check->num_rows > 0) { $visto[$chasis] = true; $resumen['duplicados']++; continue; }

    $id_estado = $id_estado_def;
    $cantidad  = $cantidad_def;
    $id_ubic   = $id_ubic_def;

    // Valores base (mismo orden de $colsBase)
    $vals = [
      $id_daientrada,
      $item, $descripcion, $modelo, $marca, $chasis, $motor, $color,
      $id_solicitudprevia, $id_estado, $cantidad, $condicion, $id_ubic, $bodega
    ];

    // Opcionales
    foreach ($optMap as $c => $_t) {
      if (!$has[$c]) continue;
      switch ($c) {
        case 'otras':        $vals[] = $ultimos4; break;
        case 'adiciones':    $vals[] = $adicionesX; break;
        case 'fobunitario':  $vals[] = $fobunit; break;
        case 'fobitem':      $vals[] = $fobitem; break;
        case 'transporte':   $vals[] = $transporte; break;
        case 'seguro':       $vals[] = $seguro; break;
        case 'valoraduana':  $vals[] = $valoraduana; break;
        case 'dai':          $vals[] = $regimen; break;
      }
    }
    if ($has['created_at']) { $vals[] = date('Y-m-d H:i:s'); }

    $stmtIns->bind_param($types, ...$vals);
    if ($stmtIns->execute()) {
      $resumen['insertados']++;
      $visto[$chasis] = true;
    } else {
      $resumen['errores']++;
    }
  }

  $stmt_check->close();
  $stmtIns->close();

  /* ---- 5) Bitácora ---- */
  if ($stmt_log = $conn->prepare(
    "INSERT INTO tb_bitacora_importaciones (nombre_archivo, id_usuario, id_solicitudprevia, id_daientrada, fecha)
     VALUES (?,?,?,?, NOW())"
  )) {
    $stmt_log->bind_param("siii", $nombre_archivo, $id_usuario, $id_solicitudprevia, $id_daientrada);
    $stmt_log->execute();
    $stmt_log->close();
  }

  $conn->commit();

  echo json_encode([
    'success' => true,
    'insertados' => $resumen['insertados'],
    'duplicados' => $resumen['duplicados'],
    'errores' => $resumen['errores'],
    'id_solicitudprevia' => $id_solicitudprevia,
    'id_daientrada' => $id_daientrada
  ]);
  exit;

} catch (Throwable $e) {
  $conn->rollback();
  echo json_encode(['success'=>false,'message'=>'Error en transacción: '.$e->getMessage()]);
  exit;
}
