<?php
declare(strict_types=1);                         // Tipos estrictos
session_start();                                 // Sesión para obtener id_usuario si existe
date_default_timezone_set('America/Guayaquil');  // Zona horaria de trabajo

/* ===== Respuesta JSON + CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }  // Preflight CORS
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {                                       // Solo POST
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Método no permitido']); exit;
}

/* ===== Conexión BD ===== */
require __DIR__ . '/../conex.php';               // Debe definir $conn (mysqli)
if (!isset($conn) || !$conn instanceof mysqli) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Sin conexión a BD']); exit;
}
mysqli_set_charset($conn, 'utf8mb4');           // Charset seguro
if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }

/* ===== Helpers ===== */
// Convierte valor a 1, 0 o NULL para AE/AI
function tri(array $src, string $key): ?int {
  if (!array_key_exists($key, $src)) return null;
  $v = $src[$key];
  if ($v === null) return null;
  if (is_bool($v)) return $v ? 1 : 0;
  if (is_int($v))  return ($v===1 ? 1 : ($v===0 ? 0 : null));
  if (is_float($v)) { $i=(int)$v; return $i===1?1:($i===0?0:null); }
  if (is_string($v)) {
    $vv = strtolower(trim($v));
    if ($vv==='' || $vv==='null' || $vv==='nulo' || $vv==='none') return null;
    if (in_array($vv, ['1','true','t','on','yes','y','si','sí'], true)) return 1;
    if (in_array($vv, ['0','false','f','off','no','n'], true))         return 0;
    if (is_numeric($vv)) { $i=(int)$vv; return $i===1?1:($i===0?0:null); }
  }
  return null;
}

/* ===== Entrada (JSON o x-www-form-urlencoded) ===== */
$raw = file_get_contents('php://input');
$in  = json_decode($raw, true);
if (!is_array($in) || !$in) { $in = $_POST ?: []; }

$id_origen     = isset($in['id_origen'])     ? (int)$in['id_origen']     : 0;    // Requerido
$id_regimen    = isset($in['id_regimen'])    ? (int)$in['id_regimen']    : null; // Opcional
$id_datadai    = isset($in['id_datadai'])    ? (int)$in['id_datadai']    : null; // Opcional
$id_inspeccion = isset($in['id_inspeccion']) ? (int)$in['id_inspeccion'] : null; // Opcional
{id_destino}   = isset($in['id_destino'])    ? (int)$in['id_destino']    : null; // Opcional
$coordinador   = isset($in['coordinador'])   ? trim((string)$in['coordinador']) : '';
$fechatarja    = isset($in['fechatarja'])    ? trim((string)$in['fechatarja'])  : date('Y-m-d');

$observacion   = isset($in['observacion'])   ? trim((string)$in['observacion']) : '';
$id_tag        = isset($in['id_tag'])        ? (int)$in['id_tag']               : 0;
$sin_tag       = isset($in['sin_tag'])       ? (int)!!$in['sin_tag']            : 0;

// AE/AI tri-estado (1, 0 o NULL)
for ($i=1; $i<=15; $i++) { ${"ae$i"} = tri($in, "ae$i"); }
for ($i=1; $i<=14; $i++) { ${"ai$i"} = tri($in, "ai$i"); }

// Usuario desde sesión o payload
$id_usuario = isset($_SESSION['idusuario']) ? (int)$_SESSION['idusuario']
           : (isset($in['id_usuario']) ? (int)$in['id_usuario'] : 0);

// Validación mínima
if ($id_origen <= 0) { http_response_code(422); echo json_encode(['success'=>false,'message'=>'id_origen es requerido']); exit; }

/* ===== Constantes de auditoría ===== */
$actividad    = 'GENERACIÓN DE TARJA ENTREGA-RECEPCIÓN VEHÍCULAR';
$ip           = $_SERVER['REMOTE_ADDR'] ?? '';
$fecha_actual = date('Y-m-d');
$hora_actual  = date('H:i:s');
$condicion    = 1;  // tarja activa

try {
  $conn->begin_transaction();                                   // Inicia transacción

  /* 1) Obtener sigla del origen (activo) */
  $stmt = $conn->prepare('SELECT sigla FROM tb_origen WHERE id = ? AND condicion = 1');
  $stmt->bind_param('i', $id_origen);
  $stmt->execute();
  $row = $stmt->get_result()->fetch_assoc();
  if (!$row) { throw new RuntimeException('Origen no válido o inactivo', 422); }
  $sigla = (string)$row['sigla'];                                // Sigla del origen (p.ej. "GU")

  /* 2) Preparar INSERT en tb_tarja (sin bloquear tablas) */
  $sqlInsertTarja = 'INSERT INTO tb_tarja
    (siglas, numero_auto, numerotarja, fechatarja, coordinador,
     id_origen, id_regimen, id_datadai, id_inspeccion, id_destino,
     ae1, ae2, ae3, ae4, ae5, ae6, ae7, ae8, ae9, ae10, ae11, ae12, ae13, ae14, ae15,
     ai1, ai2, ai3, ai4, ai5, ai6, ai7, ai8, ai9, ai10, ai11, ai12, ai13, ai14,
     observacion, id_tag, sin_tag, condicion)
   VALUES
    (?, ?, ?, ?, ?,
     ?, ?, ?, ?, ?,
     ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?,
     ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?,
     ?, ?, ?, ?)';
  $stmtIns = $conn->prepare($sqlInsertTarja);

  // Tipos:  'sisss' + 34 'i' (5 ids + 15 AE + 14 AI) + 's' + 'iii'
  $types = 'sisss' . str_repeat('i', 34) . 's' . 'iii';

  /* 3) Intentar generar correlativo y guardar (evita colisión con UNIQUE) */
  $maxReintentos = 10;                                       // Hasta 10 intentos en concurrencia
  $id_tarja      = null;
  $siglas        = 'D' . $sigla;                             // Prefijo D + sigla de origen (ej. DGU)
  $numero_auto   = 0;
  $numerotarja   = '';

  for ($try=1; $try <= $maxReintentos; $try++) {

    // Tomar candidato: último numero_auto + 1 (puede colisionar en concurrencia)
    $res = $conn->query('SELECT COALESCE(MAX(numero_auto), 0) AS maxnum FROM tb_tarja');
    $numero_auto = (int)$res->fetch_assoc()['maxnum'] + 1;

    // Armar numerotarja (ej. DGU-00012)
    $numerotarja = $siglas . '-' . str_pad((string)$numero_auto, 5, '0', STR_PAD_LEFT);

    // Bind de valores (nulls incluidos para AE/AI)
    $stmtIns->bind_param(
      $types,
      $siglas, $numero_auto, $numerotarja, $fechatarja, $coordinador,
      $id_origen, $id_regimen, $id_datadai, $id_inspeccion, $id_destino,
      $ae1, $ae2, $ae3, $ae4, $ae5, $ae6, $ae7, $ae8, $ae9, $ae10, $ae11, $ae12, $ae13, $ae14, $ae15,
      $ai1, $ai2, $ai3, $ai4, $ai5, $ai6, $ai7, $ai8, $ai9, $ai10, $ai11, $ai12, $ai13, $ai14,
      $observacion, $id_tag, $sin_tag, $condicion
    );

    try {
      $stmtIns->execute();                           // Intento de INSERT
      $id_tarja = (int)$conn->insert_id;            // Éxito -> tenemos id
      break;                                        // Salir del bucle
    } catch (mysqli_sql_exception $e) {
      if ($e->getCode() === 1062) {                 // 1062 = clave duplicada (uq_numerotarja)
        // Otro proceso insertó el mismo numerotarja: reintentar con el siguiente
        if ($try === $maxReintentos) { throw new RuntimeException('Colisión de numeración, intente de nuevo', 409); }
        continue;                                    // Siguiente vuelta del for
      }
      throw $e;                                      // Otro error SQL -> propagar
    }
  }

  /* 4) Auditoría de la TARJA */
  $stmtAud = $conn->prepare('INSERT INTO tb_auditoria
    (fecha, hora, ip, detalle, id_tarja, id_usuario, id_datadai)
    VALUES (?, ?, ?, ?, ?, ?, ?)');
  $stmtAud->bind_param('ssssiii', $fecha_actual, $hora_actual, $ip, $actividad, $id_tarja, $id_usuario, $id_datadai);
  $stmtAud->execute();

  /* 5) (Opcional) múltiples INCIDENTES */
  // Acepta 'incidentes: []' o 'incidente: {}' (compatibilidad)
  $incidentes = [];
  if (isset($in['incidentes']) && is_array($in['incidentes'])) { $incidentes = $in['incidentes']; }
  elseif (isset($in['incidente']) && is_array($in['incidente'])) { $incidentes = [ $in['incidente'] ]; }

  $incidente_ids = [];
  if (!empty($incidentes)) {
    $stmtInc = $conn->prepare('INSERT INTO tb_incidentes
      (id_zona, id_parte, id_novedad, id_tarja, id_datadai,
       observacion, id_medida, medida, foto, responsabilidad, validar, condicion)
      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1)');
    $stmtAudInc = $conn->prepare('INSERT INTO tb_auditoria
      (fecha, hora, ip, detalle, id_datadai, id_tarja, id_incidente, id_usuario)
      VALUES (?, ?, ?, ?, ?, ?, ?, ?)');

    foreach ($incidentes as $inc) {
      $id_zona      = isset($inc['id_zona'])      ? (int)$inc['id_zona']      : null;
      $id_parte     = isset($inc['id_parte'])     ? (int)$inc['id_parte']     : null;
      $id_novedad   = isset($inc['id_novedad'])   ? (int)$inc['id_novedad']   : null;
      $obs_i        = isset($inc['observacion'])  ? (string)$inc['observacion'] : null;
      $id_medida    = isset($inc['id_medida'])    ? (int)$inc['id_medida']    : null;
      $medida       = isset($inc['medida'])       ? (string)$inc['medida']    : null;
      $foto         = isset($inc['foto'])         ? (string)$inc['foto']      : null;
      $responsab    = isset($inc['responsabilidad']) ? (string)$inc['responsabilidad'] : null;
      $validar      = isset($inc['validar'])      ? (int)$inc['validar']      : 0;
      $id_datadai_i = isset($inc['id_datadai'])   ? (int)$inc['id_datadai']   : $id_datadai;

      // Insert del incidente
      $stmtInc->bind_param('iiiiisisssi',
        $id_zona, $id_parte, $id_novedad, $id_tarja, $id_datadai_i,
        $obs_i, $id_medida, $medida, $foto, $responsab, $validar
      );
      $stmtInc->execute();
      $id_incidente = (int)$conn->insert_id;
      $incidente_ids[] = $id_incidente;

      // Auditoría del incidente
      $stmtAudInc->bind_param('ssssiiii', $fecha_actual, $hora_actual, $ip, $actividad, $id_datadai_i, $id_tarja, $id_incidente, $id_usuario);
      $stmtAudInc->execute();
    }
  }

  $conn->commit();      // Confirmar todo

  /* ===== Respuesta ===== */
  echo json_encode([
    'success' => true,
    'message' => 'Tarja generada' . (!empty($incidente_ids) ? ' con '.count($incidente_ids).' incidente(s)' : ''),
    'data' => [
      'id_tarja'    => $id_tarja,
      'siglas'      => $siglas,
      'numero_auto' => $numero_auto,
      'numerotarja' => $numerotarja,
      'fechatarja'  => $fechatarja,
      'id_origen'   => $id_origen,
      'observacion' => $observacion,
      'id_tag'      => $id_tag,
      'sin_tag'     => $sin_tag,
      'ae' => [$ae1,$ae2,$ae3,$ae4,$ae5,$ae6,$ae7,$ae8,$ae9,$ae10,$ae11,$ae12,$ae13,$ae14,$ae15],
      'ai' => [$ai1,$ai2,$ai3,$ai4,$ai5,$ai6,$ai7,$ai8,$ai9,$ai10,$ai11,$ai12,$ai13,$ai14],
      'incidente_ids' => $incidente_ids
    ]
  ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  $conn->rollback();                                   // Revertir transacción
  $code = ($e instanceof RuntimeException && $e->getCode() >= 400) ? $e->getCode() : 500;
  http_response_code($code);
  echo json_encode(['success'=>false,'message'=>$e->getMessage()], JSON_UNESCAPED_UNICODE);
}
