<?php
// api/parametro_voltaje.php
declare(strict_types=1);
session_start();

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { exit; }

// ⬇️ Ajusta esta ruta según tu estructura
require_once __DIR__ . '/../conex.php';

// Soporta $conn o $conex
$db = $conn ?? ($conex ?? null);
if (!$db) {
  http_response_code(500);
  echo json_encode(['success' => false, 'message' => 'Sin conexión a la base de datos']);
  exit;
}

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone = '-05:00'");

/* ========= Entrada =========
   - valor: voltaje (float)  (obligatorio)
   - id_empresa: int         (obligatorio; si no hay coincidencia, se usa fallback)
   - unidad: string          (opcional, por defecto 'Voltaje')
*/
$input = $_POST + $_GET;

// Permitir JSON crudo también
if (empty($input)) {
  $raw = file_get_contents('php://input');
  if ($raw) {
    $json = json_decode($raw, true);
    if (is_array($json)) { $input = $json; }
  }
}

$valor      = isset($input['valor']) ? floatval($input['valor']) : null;
$idEmpresa  = isset($input['id_empresa']) ? intval($input['id_empresa']) : null;
$unidad     = trim((string)($input['unidad'] ?? 'Voltaje'));

if ($valor === null || $idEmpresa === null || $unidad === '') {
  http_response_code(400);
  echo json_encode([
    'success' => false,
    'message' => 'Parámetros requeridos: valor (float), id_empresa (int). Opcional: unidad (string).'
  ]);
  exit;
}

/*
  Consulta única con Fallback:
  1) Busca coincidencias en la empresa indicada.
  2) Si no hay, trae de otras empresas (NOT EXISTS del primer set).
  Nota: usamos comparaciones explícitas en lugar de BETWEEN para poder parametrizar el valor.
*/
$sql = "
(
  SELECT p.accion, p.tiempo
  FROM tb_parametros AS p
  WHERE p.condicion = 1
    AND p.unidad = ?
    AND p.id_empresa = ?
    AND (
         (? >= p.ninimo AND ? <= p.maximo)
         OR (p.maximo = 0 AND ? >= p.ninimo)   -- sin límite superior
         OR (p.ninimo = 0 AND ? <= p.maximo)   -- sin límite inferior
    )
)
UNION ALL
(
  SELECT p.accion, p.tiempo
  FROM tb_parametros AS p
  WHERE p.condicion = 1
    AND p.unidad = ?
    AND p.id_empresa <> ?
    AND (
         (? >= p.ninimo AND ? <= p.maximo)
         OR (p.maximo = 0 AND ? >= p.ninimo)
         OR (p.ninimo = 0 AND ? <= p.maximo)
    )
    AND NOT EXISTS (
      SELECT 1
      FROM tb_parametros AS x
      WHERE x.condicion = 1
        AND x.unidad = ?
        AND x.id_empresa = ?
        AND (
             (? >= x.ninimo AND ? <= x.maximo)
             OR (x.maximo = 0 AND ? >= x.ninimo)
             OR (x.ninimo = 0 AND ? <= x.maximo)
        )
    )
)
";

$stmt = $db->prepare($sql);
if (!$stmt) {
  http_response_code(500);
  echo json_encode([
    'success' => false,
    'message' => 'Error al preparar la consulta',
    'error'   => mysqli_error($db)
  ]);
  exit;
}

/*
  Orden de parámetros (tipos):
  1) Primer bloque: unidad (s), id_empresa (i), valor (d), valor (d), valor (d), valor (d)
  2) Segundo bloque: unidad (s), id_empresa (i), valor (d), valor (d), valor (d), valor (d),
     unidad (s), id_empresa (i), valor (d), valor (d), valor (d), valor (d)
  Total: 18 parámetros
*/
$types = 'si dddd s i dddd s i dddd'; // separados por espacios para leer; abajo quitamos espacios
$types = str_replace(' ', '', $types);

// Construimos la lista de valores en el orden del SQL
$params = [
  // 1) Primer bloque
  $unidad, $idEmpresa, $valor, $valor, $valor, $valor,
  // 2) Segundo bloque (select 2)
  $unidad, $idEmpresa, $valor, $valor, $valor, $valor,
  //    NOT EXISTS (subconsulta)
  $unidad, $idEmpresa, $valor, $valor, $valor, $valor
];

// Vincular parámetros dinámicamente
$stmt->bind_param($types, ...$params);

if (!$stmt->execute()) {
  http_response_code(500);
  echo json_encode([
    'success' => false,
    'message' => 'Error al ejecutar la consulta',
    'error'   => $stmt->error
  ]);
  $stmt->close();
  exit;
}

$res = $stmt->get_result();
$data = [];
while ($row = $res->fetch_assoc()) {
  // Normaliza claves si deseas
  $data[] = [
    'accion' => $row['accion'],
    'tiempo' => $row['tiempo'],
  ];
}
$stmt->close();

echo json_encode([
  'success' => true,
  'unidad' => $unidad,
  'valor' => $valor,
  'id_empresa' => $idEmpresa,
  'rows' => count($data),
  'data' => $data
], JSON_UNESCAPED_UNICODE);
