<?php
// Devuelve: vehiculo (marca, modelo, chasis, color, dias_transcurridos, fecha_ultimo_mantenimiento)
// + tipos que tocan según regla exacta indicada.

declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }

date_default_timezone_set('America/Guayaquil');

/* Conexión */
$pathConex = realpath(__DIR__ . '/../conex.php')
          ?: realpath(__DIR__ . '/../../conex.php')
          ?: realpath(__DIR__ . '/conex.php');
if (!$pathConex) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'No se encontró conex.php']); exit; }
require_once $pathConex;

$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'Sin conexión a la base de datos']); exit; }
mysqli_set_charset($db, 'utf8mb4');

/* Entrada (GET/POST JSON/FORM) */
$inJson = [];
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'POST' && stripos($_SERVER['CONTENT_TYPE'] ?? '', 'application/json') !== false) {
  $raw = file_get_contents('php://input'); $inJson = json_decode($raw, true) ?: [];
}
$chasis = trim((string)($_GET['chasis'] ?? $inJson['chasis'] ?? $_POST['chasis'] ?? ''));
if ($chasis === '') { http_response_code(400); echo json_encode(['success'=>false,'message'=>'Parámetro "chasis" es requerido']); exit; }

/* 1) Vehículo */
$sqlVeh = "SELECT iddatadai, marca, modelo, chasis, color, created_at
           FROM tb_datadai
           WHERE UPPER(chasis)=UPPER(?)
           LIMIT 1";
$stmtVeh = $db->prepare($sqlVeh);
if (!$stmtVeh) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'Error preparando consulta de vehículo']); exit; }
$stmtVeh->bind_param('s', $chasis);
$stmtVeh->execute();
$resVeh = $stmtVeh->get_result();
$veh = $resVeh->fetch_assoc();
$stmtVeh->close();
if (!$veh) { http_response_code(404); echo json_encode(['success'=>false,'message'=>'No se encontró un vehículo con ese chasis']); exit; }

$vehId   = (int)$veh['iddatadai'];
$marca   = (string)($veh['marca']  ?? '');
$modelo  = (string)($veh['modelo'] ?? '');
$chas    = (string)($veh['chasis'] ?? '');
$color   = (string)($veh['color']  ?? '');
$createdRaw = $veh['created_at'] ?? null;

/* 2) Último mantenimiento (BASE preferida) */
$sqlUlt = "SELECT MAX(fecha) AS ultima_fecha
           FROM tb_mantenimiento
           WHERE id_datadai=? AND (condicion=1 OR condicion IS NULL)";
$stmtUlt = $db->prepare($sqlUlt);
$ultimaFecha = null; // string|NULL
if ($stmtUlt) {
  $stmtUlt->bind_param('i', $vehId);
  $stmtUlt->execute();
  $resUlt = $stmtUlt->get_result();
  $rowUlt = $resUlt->fetch_assoc();
  $ultimaFecha = $rowUlt['ultima_fecha'] ?? null; // p.ej. '2025-09-20 14:10:00'
  $stmtUlt->close();
}

/* 3) Días transcurridos: BASE = último mantenimiento o created_at */
$baseRaw = $ultimaFecha ?: $createdRaw;
$diasTranscurridos = 0;
if ($baseRaw) {
  try {
    $base = new DateTime($baseRaw);
    $hoy  = new DateTime('today');
    $diasTranscurridos = (int)$base->diff($hoy)->days;
  } catch (Throwable $e) { $diasTranscurridos = 0; }
}

/* 4) Regla exacta */
$solo30    = [30,60,120,150,210,240,300,330,390,410,420,480,510];
$treinta90 = [90,180,270,450];
$pack360   = [360];
$pack540   = [540];

$necesariosDias = [];
if (in_array($diasTranscurridos, $solo30, true)) {
  $necesariosDias = [30];
} elseif (in_array($diasTranscurridos, $treinta90, true)) {
  $necesariosDias = [30, 90];
} elseif (in_array($diasTranscurridos, $pack360, true)) {
  $necesariosDias = [30, 90, 360];
} elseif (in_array($diasTranscurridos, $pack540, true)) {
  $necesariosDias = [30, 90, 360, 540];
}

/* 5) Cargar desde la tabla SOLO esos tipos (activos) */
$tipos = [];
if ($necesariosDias) {
  $placeholders = implode(',', array_fill(0, count($necesariosDias), '?'));
  $types = str_repeat('i', count($necesariosDias));
  $sqlTipos = "SELECT id, mantenimiento, dias
               FROM tb_tipo_mantenimiento
               WHERE condicion=1 AND dias IN ($placeholders)
               ORDER BY dias ASC";
  $stmt = $db->prepare($sqlTipos);
  if ($stmt) {
    $stmt->bind_param($types, ...$necesariosDias);
    $stmt->execute();
    $res = $stmt->get_result();
    while ($r = $res->fetch_assoc()) {
      $tipos[] = [
        'id'            => (int)$r['id'],
        'mantenimiento' => (string)$r['mantenimiento'],
        'dias'          => (int)$r['dias']
      ];
    }
    $stmt->close();
  }
}

/* 6) Respuesta */
echo json_encode([
  'success' => true,
  'vehiculo'=> [
    'marca' => $marca,
    'modelo'=> $modelo,
    'chasis'=> $chas,
    'color' => $color,
    'dias_transcurridos' => $diasTranscurridos,
    'fecha_ultimo_mantenimiento' => $ultimaFecha ?: null
  ],
  'tipos'   => $tipos // solo los que tocan
], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
