<?php
// views/pages/api/requerimiento_guardar.php — Inserta requerimientos en tb_pdi (1 fila por vehículo) + LUGAR
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

session_start();
if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES') {
  http_response_code(401);
  echo json_encode(['success'=>false,'message'=>'No autorizado']); exit;
}
$usuario = (string)($_SESSION['usuario'] ?? 'sistema');

require_once __DIR__ . '/../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { echo json_encode(['success'=>false,'message'=>'Sin conexión BD']); exit; }

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone = '-05:00'"); // America/Guayaquil

// ===== Helpers =====
function is_valid_dt(string $dt): bool {
  $dt = str_replace('T', ' ', $dt);
  $d = DateTime::createFromFormat('Y-m-d H:i:s', $dt);
  return $d && $d->format('Y-m-d H:i:s') === $dt;
}
function norm_dt(string $dt): string { return str_replace('T', ' ', $dt); }

// ===== Entrada =====
$raw = file_get_contents('php://input');
$in  = json_decode($raw, true) ?: [];

$h  = $in['header']    ?? [];
$vs = $in['vehiculos'] ?? [];

// Header
$alistamiento = trim((string)($h['alistamiento'] ?? ''));
$requerido    = trim((string)($h['requerido']    ?? ''));
$lugar        = trim((string)($h['lugar']        ?? ''));
$requerimDT   = norm_dt(trim((string)($h['requerimiento']?? ''))); // 'YYYY-mm-dd HH:ii:ss'
$entregaDT    = norm_dt(trim((string)($h['entrega']      ?? '')));
$prioridad    = strtoupper(trim((string)($h['prioridad'] ?? 'NORMAL')));
$facturado    = trim((string)($h['facturado']    ?? ''));

// ===== Validaciones =====
if (!in_array($alistamiento, ['Con PDI','Sin PDI'], true)) {
  echo json_encode(['success'=>false,'message'=>'Tipo de alistamiento inválido (Con PDI / Sin PDI)']); exit;
}
if ($prioridad !== 'NORMAL' && $prioridad !== 'URGENTE') {
  echo json_encode(['success'=>false,'message'=>'Prioridad inválida (NORMAL / URGENTE)']); exit;
}
if ($requerido === '' || $lugar === '' || $requerimDT === '' || $entregaDT === '' || $facturado === '') {
  echo json_encode(['success'=>false,'message'=>'Faltan campos obligatorios']); exit;
}
if (!is_valid_dt($requerimDT) || !is_valid_dt($entregaDT)) {
  echo json_encode(['success'=>false,'message'=>'Formato de fecha/hora inválido. Use YYYY-mm-dd HH:ii:ss']); exit;
}
if (strtotime($entregaDT) < strtotime($requerimDT)) {
  echo json_encode(['success'=>false,'message'=>'La fecha de entrega no puede ser menor a la de requerimiento']); exit;
}
if (!is_array($vs) || count($vs) === 0) {
  echo json_encode(['success'=>false,'message'=>'Sin vehículos']); exit;
}

// Normalizar vehículos: iddatadai únicos válidos (>0)
$vehiculos = [];
foreach ($vs as $v) {
  $idd = (int)($v['iddatadai'] ?? 0);
  if ($idd > 0) $vehiculos[] = $idd;
}
$vehiculos = array_values(array_unique($vehiculos));
if (!$vehiculos) { echo json_encode(['success'=>false,'message'=>'Vehículos inválidos']); exit; }

try {
  $db->begin_transaction();

  // Validar que existan en tb_datadai
  $ph    = implode(',', array_fill(0, count($vehiculos), '?'));
  $types = str_repeat('i', count($vehiculos));
  $stmtV = $db->prepare("SELECT iddatadai FROM tb_datadai WHERE iddatadai IN ($ph)");
  if (!$stmtV) { throw new RuntimeException('Prep validación tb_datadai: '.$db->error); }
  $stmtV->bind_param($types, ...$vehiculos);
  $stmtV->execute();
  $rsV = $stmtV->get_result();
  $valid = [];
  while ($r = $rsV->fetch_assoc()) { $valid[(int)$r['iddatadai']] = true; }
  $stmtV->close();

  $insertados = 0; $saltados = 0; $noEncontrados = 0; $duplicados = 0;

  // Duplicado por (id_datadai, fecha_req) con condicion=1
  $stmtDupe = $db->prepare("SELECT COUNT(*) c FROM tb_pdi WHERE id_datadai = ? AND fecha_req = ? AND condicion = 1");
  if (!$stmtDupe) { throw new RuntimeException('Prep verificación duplicado: '.$db->error); }

  // INSERT con LUGAR (nuevo campo)
  // Columnas: id_datadai, alistamiento, fecha_req, fecha_ent, requerido_por, lugar, condicion, prioridad, facturado, fecha_reg
  $stmtIns = $db->prepare("
    INSERT INTO tb_pdi
      (id_datadai, alistamiento, fecha_req, fecha_ent, requerido_por, lugar, condicion, prioridad, facturado, fecha_reg)
    VALUES
      (?,?,?,?,?,?,1,?, ?, NOW())
  ");
  if (!$stmtIns) { throw new RuntimeException('Prep insert: '.$db->error); }

  foreach ($vehiculos as $idd) {
    if (empty($valid[$idd])) { $noEncontrados++; continue; }

    // chequear duplicado
    $stmtDupe->bind_param('is', $idd, $requerimDT);
    $stmtDupe->execute();
    $dupeRow = $stmtDupe->get_result()->fetch_assoc();
    $c = (int)($dupeRow['c'] ?? 0);
    if ($c > 0) { $duplicados++; $saltados++; continue; }

    // bind: i s s s s s s s  (8 parámetros antes de NOW())
    $stmtIns->bind_param(
      'isssssss',
      $idd,            // id_datadai
      $alistamiento,   // alistamiento
      $requerimDT,     // fecha_req
      $entregaDT,      // fecha_ent
      $requerido,      // requerido_por
      $lugar,          // lugar   <<< NUEVO
      $prioridad,      // prioridad
      $facturado       // facturado
    );

    if (!$stmtIns->execute()) { throw new RuntimeException('Insert falló: '.$db->error); }
    $insertados++;
  }

  $stmtDupe->close();
  $stmtIns->close();

  $db->commit();
  echo json_encode([
    'success'        => true,
    'insertados'     => $insertados,
    'saltados'       => $saltados,
    'duplicados'     => $duplicados,
    'no_encontrados' => $noEncontrados,
    'message'        => 'Proceso finalizado'
  ], JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
  $db->rollback();
  http_response_code(500);
  echo json_encode(['success'=>false, 'message'=>'Transacción fallida', 'sql_error'=>$e->getMessage()]);
}
