<?php
// views/pages/api/novedades_chasis_api.php
// API JSON para listar novedades por chasis (modo simple y DataTables server-side) — SIN Tarja en columnas
declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

require_once __DIR__ . '/../conex.php';
if (!$conn instanceof mysqli) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Error de conexión a BD']); exit;
}
mysqli_set_charset($conn, 'utf8mb4');
if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }

function qp(string $k, $d=null){ return $_GET[$k] ?? $d; }
function like_esc(string $s): string { return strtr($s, ['\\'=>'\\\\', '%'=>'\\%', '_'=>'\\_']); }

/** Filtros por columna (índices ya SIN Tarja): 
 * 0 chasis
 * 1 marca/modelo/color
 * 2 zona
 * 3 parte
 * 4 novedad
 * 5 medida
 * 6 observacion
 * 7 reponsabilidad
 * 8 validar
 * 9 grave
 * 10 foto
 */
function apply_col_filter(int $idx, string $val, bool $isRegex, array &$where, array &$params, string &$types): void {
  if ($val === '') return;

  switch ($idx) {
    case 0:
      $where[]="d.chasis LIKE ? ESCAPE '\\\\'"; $params[]='%'.like_esc($val).'%'; $types.='s'; break;
    case 1:
      $like='%'.like_esc($val).'%';
      $where[]="(d.marca LIKE ? ESCAPE '\\\\' OR d.modelo LIKE ? ESCAPE '\\\\' OR d.color LIKE ? ESCAPE '\\\\')";
      $params[]=$like; $types.='s'; $params[]=$like; $types.='s'; $params[]=$like; $types.='s'; break;
    case 2:
      $where[]="z.detallezona LIKE ? ESCAPE '\\\\'"; $params[]='%'.like_esc($val).'%'; $types.='s'; break;
    case 3:
      $where[]="p.detalleparte LIKE ? ESCAPE '\\\\'"; $params[]='%'.like_esc($val).'%'; $types.='s'; break;
    case 4:
      $where[]="n.detallenovedad LIKE ? ESCAPE '\\\\'"; $params[]='%'.like_esc($val).'%'; $types.='s'; break;
    case 5:
      $where[]="i.medida LIKE ? ESCAPE '\\\\'"; $params[]='%'.like_esc($val).'%'; $types.='s'; break;
    case 6:
      $where[]="i.observacion LIKE ? ESCAPE '\\\\'"; $params[]='%'.like_esc($val).'%'; $types.='s'; break;
    case 7: // reponsabilidad 0/1 exacto
      if ($isRegex && preg_match('/^\^([01])\$$/', $val, $m)) {
        $where[]="i.reponsabilidad = ?"; $params[]=(int)$m[1]; $types.='i';
      } else {
        $where[]="CAST(i.reponsabilidad AS CHAR) LIKE ? ESCAPE '\\\\'"; $params[]='%'.like_esc($val).'%'; $types.='s';
      }
      break;
    case 8: // validar 0/1 exacto
      if ($isRegex && preg_match('/^\^([01])\$$/', $val, $m)) {
        $where[]="i.validar = ?"; $params[]=(int)$m[1]; $types.='i';
      } else {
        $where[]="CAST(i.validar AS CHAR) LIKE ? ESCAPE '\\\\'"; $params[]='%'.like_esc($val).'%'; $types.='s';
      }
      break;
    case 9: // grave 0/1 exacto
      if ($isRegex && preg_match('/^\^([01])\$$/', $val, $m)) {
        $where[]="i.grave = ?"; $params[]=(int)$m[1]; $types.='i';
      } else {
        $where[]="CAST(i.grave AS CHAR) LIKE ? ESCAPE '\\\\'"; $params[]='%'.like_esc($val).'%'; $types.='s';
      }
      break;
    case 10:
      $where[]="i.foto LIKE ? ESCAPE '\\\\'"; $params[]='%'.like_esc($val).'%'; $types.='s'; break;
  }
}

try{
  if (qp('ajax','') !== '1' && qp('mode','') !== 'dt') {
    echo json_encode(['success'=>false,'message'=>'Uso inválido']); exit;
  }

  // Filtros globales
  $search     = trim((string)qp('search',''));
  $grave      = qp('grave','');     // '', '0', '1'
  $validar    = qp('validar','');   // '', '0', '1'
  $condicion  = qp('condicion',''); // '', '0', '1'
  $chasisList = trim((string)qp('chasis_list',''));
  $mode       = qp('mode','');      // '' simple, 'dt' server-side

  $lista = [];
  if ($chasisList !== '') {
    foreach (preg_split('/[,\s]+/', $chasisList) as $c) { $c=trim($c); if($c!=='') $lista[]=$c; }
    $lista = array_values(array_unique($lista));
  }

  $where=[]; $params=[]; $types='';

  if (!empty($lista)) {
    $in = implode(',', array_fill(0, count($lista), '?'));
    $where[] = "d.chasis IN ($in)";
    foreach ($lista as $c) { $params[]=$c; $types.='s'; }
  }

  if ($search !== '') {
    $like = '%'.like_esc($search).'%';
    $where[] = "(d.chasis LIKE ? ESCAPE '\\\\'
                OR d.marca LIKE ? ESCAPE '\\\\'
                OR d.modelo LIKE ? ESCAPE '\\\\'
                OR d.color  LIKE ? ESCAPE '\\\\'
                OR i.observacion LIKE ? ESCAPE '\\\\'
                OR n.detallenovedad LIKE ? ESCAPE '\\\\'
                OR p.detalleparte LIKE ? ESCAPE '\\\\'
                OR z.detallezona  LIKE ? ESCAPE '\\\\')";
    for ($i=0; $i<8; $i++) { $params[]=$like; $types.='s'; }
  }

  if ($grave === '0' || $grave === '1')        { $where[]="i.grave=?";      $params[]=(int)$grave;     $types.='i'; }
  if ($validar === '0' || $validar === '1')    { $where[]="i.validar=?";    $params[]=(int)$validar;   $types.='i'; }
  if ($condicion === '0' || $condicion === '1'){ $where[]="i.condicion=?";  $params[]=(int)$condicion; $types.='i'; }

  $SQL_BASE = "
    FROM tb_incidente i
    INNER JOIN tb_datadai d ON d.iddatadai = i.id_datadai
    LEFT JOIN tb_zona   z ON z.idzona  = i.id_zona
    LEFT JOIN tb_parte  p ON p.idparte = i.id_parte
    LEFT JOIN tb_novedad n ON n.idnovedad = i.id_novedad
  ";

  $SQL_SELECT = "
    SELECT
      i.idincidente,
      d.iddatadai, d.chasis, d.marca, d.modelo, d.color,
      i.id_zona,  z.detallezona  AS zona,
      i.id_parte, p.detalleparte AS parte,
      i.id_novedad, n.detallenovedad AS novedad,
      i.id_medida, i.medida,
      i.observacion, i.foto,
      i.reponsabilidad, i.validar, i.condicion, i.grave
  ";

  if ($mode === 'dt') {
    // Filtros por columna del thead
    if (isset($_GET['columns']) && is_array($_GET['columns'])) {
      foreach ($_GET['columns'] as $idx => $col) {
        $val = isset($col['search']['value']) ? trim((string)$col['search']['value']) : '';
        $isRegex = isset($col['search']['regex']) && ($col['search']['regex'] === 'true' || $col['search']['regex'] === true);
        if ($val !== '') apply_col_filter((int)$idx, $val, (bool)$isRegex, $where, $params, $types);
      }
    }

    $whereSql = $where ? 'WHERE '.implode(' AND ', $where) : '';

    // Ordenables (SIN Tarja)
    $orderable = [
      0 => 'd.chasis',
      1 => 'd.marca',              // proxy para marca/modelo/color
      2 => 'zona',
      3 => 'parte',
      4 => 'n.detallenovedad',
      5 => 'i.medida',
      6 => 'i.observacion',
      7 => 'i.reponsabilidad',
      8 => 'i.validar',
      9 => 'i.grave',
      10 => 'i.foto'
    ];

    $draw   = (int)(qp('draw', 0));
    $start  = max(0, (int)(qp('start', 0)));
    $length = (int)(qp('length', 25)); if ($length <= 0 || $length > 1000) $length = 25;

    $orderColIdx = isset($_GET['order'][0]['column']) ? (int)$_GET['order'][0]['column'] : 0;
    $orderDir    = isset($_GET['order'][0]['dir']) ? strtolower((string)$_GET['order'][0]['dir']) : 'asc';
    $orderDir    = ($orderDir === 'desc') ? 'DESC' : 'ASC';
    $orderColSql = $orderable[$orderColIdx] ?? 'i.idincidente';

    $sqlTotal = "SELECT COUNT(*) AS c FROM tb_incidente i INNER JOIN tb_datadai d ON d.iddatadai = i.id_datadai";
    $total = (int)$conn->query($sqlTotal)->fetch_assoc()['c'];

    $sqlFiltered = "SELECT COUNT(*) AS c $SQL_BASE $whereSql";
    $stmtCount = $conn->prepare($sqlFiltered);
    if ($types !== '') { $stmtCount->bind_param($types, ...$params); }
    $stmtCount->execute();
    $resCount = $stmtCount->get_result();
    $filtered = (int)($resCount->fetch_assoc()['c'] ?? 0);
    $stmtCount->close();

    $sqlData = $SQL_SELECT . " $SQL_BASE $whereSql ORDER BY $orderColSql $orderDir LIMIT ?, ?";
    $stmt = $conn->prepare($sqlData);
    if ($types !== '') {
      $types2 = $types . 'ii';
      $params2 = array_merge($params, [$start, $length]);
      $stmt->bind_param($types2, ...$params2);
    } else {
      $stmt->bind_param('ii', $start, $length);
    }
    $stmt->execute();
    $res = $stmt->get_result();

    $data = [];
    while ($row = $res->fetch_assoc()) {
      $data[] = [
        'idincidente'     => (int)$row['idincidente'],
        'iddatadai'       => (int)$row['iddatadai'],
        'chasis'          => (string)($row['chasis'] ?? ''),
        'marca'           => (string)($row['marca'] ?? ''),
        'modelo'          => (string)($row['modelo'] ?? ''),
        'color'           => (string)($row['color'] ?? ''),
        'zona'            => (string)($row['zona'] ?? ''),
        'parte'           => (string)($row['parte'] ?? ''),
        'novedad'         => (string)($row['novedad'] ?? ''),
        'medida'          => (string)($row['medida'] ?? ''),
        'observacion'     => (string)($row['observacion'] ?? ''),
        'reponsabilidad'  => (string)($row['reponsabilidad'] ?? ''),
        'validar'         => (int)($row['validar'] ?? 0),
        'grave'           => (int)($row['grave'] ?? 0),
        'condicion'       => (int)($row['condicion'] ?? 0),
        'foto'            => (string)($row['foto'] ?? '')
      ];
    }
    $stmt->close();

    echo json_encode([
      'draw' => $draw,
      'recordsTotal' => $total,
      'recordsFiltered' => $filtered,
      'data' => $data,
      'success' => true
    ], JSON_UNESCAPED_UNICODE);
    exit;
  }

  // Modo simple (hasta 1000 filas)
  $whereSql = $where ? 'WHERE '.implode(' AND ', $where) : '';
  $sql = $SQL_SELECT . " $SQL_BASE $whereSql ORDER BY i.idincidente DESC LIMIT 1000";
  $stmt = $conn->prepare($sql);
  if ($types!=='') { $stmt->bind_param($types, ...$params); }
  $stmt->execute();
  $res = $stmt->get_result();

  $data = [];
  while ($row = $res->fetch_assoc()) {
    $data[] = [
      'idincidente'     => (int)$row['idincidente'],
      'iddatadai'       => (int)$row['iddatadai'],
      'chasis'          => (string)($row['chasis'] ?? ''),
      'marca'           => (string)($row['marca'] ?? ''),
      'modelo'          => (string)($row['modelo'] ?? ''),
      'color'           => (string)($row['color'] ?? ''),
      'zona'            => (string)($row['zona'] ?? ''),
      'parte'           => (string)($row['parte'] ?? ''),
      'novedad'         => (string)($row['novedad'] ?? ''),
      'medida'          => (string)($row['medida'] ?? ''),
      'observacion'     => (string)($row['observacion'] ?? ''),
      'reponsabilidad'  => (string)($row['reponsabilidad'] ?? ''),
      'validar'         => (int)($row['validar'] ?? 0),
      'grave'           => (int)($row['grave'] ?? 0),
      'condicion'       => (int)($row['condicion'] ?? 0),
      'foto'            => (string)($row['foto'] ?? '')
    ];
  }
  $stmt->close();

  echo json_encode(['success'=>true, 'data'=>$data], JSON_UNESCAPED_UNICODE);

} catch(Throwable $e){
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Error en la consulta','error'=>$e->getMessage()]);
}
