<?php
// api/modelo_sync.php — Sync sin duplicados con UNIQUE por nombre_norm
declare(strict_types=1);
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

session_start();
require_once __DIR__ . '/../conex.php';
if (!isset($conn) || !($conn instanceof mysqli)) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Sin conexión a BD']); exit;
}
mysqli_set_charset($conn,'utf8mb4');

$ORIGEN  = 'tb_datadai';
$MARCAS  = 'tb_inv_marca';
$MODELOS = 'tb_inv_modelo';

function sql_fail(mysqli $c, string $msg, int $code=500){
  http_response_code($code);
  echo json_encode(['success'=>false,'message'=>$msg,'sql_error'=>mysqli_error($c)]); exit;
}

try {
  mysqli_begin_transaction($conn);

  // 1) Insertar MARCAS nuevas
  $sqlMarcas = "
    INSERT INTO {$MARCAS} (nombre, condicion)
    SELECT DISTINCT TRIM(d.marca) AS nombre, 1
    FROM {$ORIGEN} d
    WHERE TRIM(d.marca) <> ''
      AND NOT EXISTS (
        SELECT 1 FROM {$MARCAS} m
        WHERE m.nombre = TRIM(d.marca)
      )
  ";
  if (!mysqli_query($conn, $sqlMarcas)) sql_fail($conn, 'Error al sincronizar marcas');
  $ins_brands = mysqli_affected_rows($conn);

  // 2) Insertar MODELOS (ignorar duplicados por UNIQUE (id_marca, nombre_norm, anio))
  // anio: solo YYYY -> entero; otro caso -> NULL
  $sqlModelos = "
    INSERT INTO {$MODELOS} (id_marca, nombre, anio, condicion)
    SELECT
      m.id_marca,
      /* nombre sin normalizar aquí; nombre_norm se genera solo al insertar */
      TRIM(d.modelo) AS nombre,
      CASE
        WHEN TRIM(d.otras) REGEXP '^[0-9]{4}$' THEN CAST(d.otras AS UNSIGNED)
        ELSE NULL
      END AS anio,
      CASE
        WHEN d.condicion IS NULL THEN 1
        WHEN d.condicion IN (0,1) THEN d.condicion
        ELSE 1
      END AS condicion
    FROM {$ORIGEN} d
    JOIN {$MARCAS} m
      ON UPPER(m.nombre) = UPPER(TRIM(d.marca))
    WHERE TRIM(d.modelo) <> ''
  ON DUPLICATE KEY UPDATE
    -- no-op (conserva el existente); si quisieras actualizar algo, hazlo aquí
    id_marca = VALUES(id_marca)
  ";
  if (!mysqli_query($conn, $sqlModelos)) sql_fail($conn, 'Error al sincronizar modelos');
  $ins_models = mysqli_affected_rows($conn); // insertados o tocados por el DUPLICATE

  mysqli_commit($conn);

  echo json_encode([
    'success'=>true,
    'message'=>'Sincronización completada',
    'inserted_brands'=>$ins_brands,
    'inserted_models'=>$ins_models
  ]);
} catch (Throwable $e) {
  mysqli_rollback($conn);
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>$e->getMessage()]);
}
