<?php
// api/accesorizacion_guardar.php
// Espera: { chasis:"ABC123", accesorios:["1","3","7"] }  (IDs de tb_accesorios)
// Lógica por chasis (id_datadai):
//   - UPSERT (INSERT ... ON DUPLICATE) para activar/crear cada accesorio seleccionado (estado=1)
//   - Desactivar (estado=0) los que estaban activos y ya no vienen en la lista
declare(strict_types=1);
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
if ($_SERVER['REQUEST_METHOD']==='OPTIONS'){ http_response_code(204); exit; }

session_start();
require_once __DIR__ . '/../conex.php';

try{
  if ($_SERVER['REQUEST_METHOD']!=='POST') throw new RuntimeException('Método no permitido');
  if (!isset($conn) || !($conn instanceof mysqli)) throw new RuntimeException('Sin conexión BD');
  mysqli_set_charset($conn,'utf8mb4');

  $in     = json_decode(file_get_contents('php://input'), true) ?: [];
  $chasis = trim((string)($in['chasis']??''));
  $arr    = $in['accesorios'] ?? [];

  if ($chasis==='') throw new InvalidArgumentException('chasis requerido');
  if (!is_array($arr)) throw new InvalidArgumentException('accesorios debe ser array');

  // Normalizar a set de enteros únicos
  $want = [];
  foreach($arr as $v){ $i=(int)$v; if($i>0) $want[$i]=true; }

  $idUsuario = (int)($_SESSION['idusuario'] ?? 0);

  // 1) Resolver id_datadai
  $q = "SELECT iddatadai FROM tb_datadai WHERE chasis=? AND condicion=1 LIMIT 1";
  $st = mysqli_prepare($conn,$q);
  mysqli_stmt_bind_param($st,'s',$chasis);
  mysqli_stmt_execute($st);
  $res = mysqli_stmt_get_result($st);
  $idd = 0;
  if ($r = mysqli_fetch_assoc($res)) $idd = (int)$r['iddatadai'];
  if ($idd<=0) throw new RuntimeException('Vehículo no encontrado');

  // 2) Leer activos actuales
  $qCur = "SELECT id_accesorios FROM tb_accesorizacion WHERE id_datadai=? AND estado_acces=1";
  $stCur = mysqli_prepare($conn,$qCur);
  mysqli_stmt_bind_param($stCur,'i',$idd);
  mysqli_stmt_execute($stCur);
  $resCur = mysqli_stmt_get_result($stCur);

  $haveActive = []; // set de accesorios activos actuales
  while($row = mysqli_fetch_assoc($resCur)){
    $haveActive[(int)$row['id_accesorios']] = true;
  }

  // 3) Diferencias
  $toActivate = [];   // vienen y no estaban activos
  foreach($want as $aid => $_){
    if (!isset($haveActive[$aid])) $toActivate[] = $aid;
  }
  $toDeactivate = []; // estaban activos y ya no vienen
  foreach($haveActive as $aid => $_){
    if (!isset($want[$aid])) $toDeactivate[] = $aid;
  }

  // 4) Aplicar cambios en transacción
  mysqli_begin_transaction($conn);

  // 4.a) Activar/crear seleccionados (UPSERT por UNIQUE (id_datadai, id_accesorios))
  if (!empty($toActivate)){
    $ins = mysqli_prepare(
      $conn,
      "INSERT INTO tb_accesorizacion (id_datadai, id_accesorios, id_usuario_a, estado_acces)
       VALUES (?, ?, ?, 1)
       ON DUPLICATE KEY UPDATE id_usuario_a=VALUES(id_usuario_a), estado_acces=1"
    );
    foreach($toActivate as $aid){
      mysqli_stmt_bind_param($ins,'iii',$idd,$aid,$idUsuario);
      mysqli_stmt_execute($ins);
    }
  }

  // 4.b) Desactivar los que se quitaron (histórico)
  if (!empty($toDeactivate)){
    // Armamos UPDATE ... IN (?, ?, ?)
    $place = implode(',', array_fill(0, count($toDeactivate), '?'));
    $sqlUp = "UPDATE tb_accesorizacion
              SET estado_acces=0, id_usuario_a=?
              WHERE id_datadai=? AND id_accesorios IN ($place)";
    $stmtUp = mysqli_prepare($conn, $sqlUp);

    // Tipos: id_usuario_a (i), id_datadai (i), luego N accesorios (i...i)
    $bindTypes = 'ii' . str_repeat('i', count($toDeactivate));
    $args = [$bindTypes, $idUsuario, $idd];
    foreach($toDeactivate as $aid){ $args[] = $aid; }

    // bind_param dinámico (por referencia)
    $refs = [];
    foreach ($args as $k => $v) { $refs[$k] = &$args[$k]; }
    call_user_func_array([$stmtUp, 'bind_param'], $refs);

    mysqli_stmt_execute($stmtUp);
  }

  mysqli_commit($conn);

  echo json_encode(['success'=>true,'message'=>'Guardado (un registro por accesorio)'], JSON_UNESCAPED_UNICODE);
}catch(Throwable $e){
  if (isset($conn) && $conn instanceof mysqli){ @mysqli_rollback($conn); }
  http_response_code(400);
  echo json_encode(['success'=>false,'message'=>$e->getMessage()]);
}
